<?php

namespace PriceReader\Data;

/**
 * Description of PriceData
 *
 * @author d.lanec
 */
class PriceData
{

    protected  $notUseConvertToUtf = false;

    /**
     * путь к файлу
     * @var string
     */
    protected $file;

    protected $count_rows = 0;

    protected $colCount = 0;

    protected $limitrows;

    protected $encoding;

    protected $data = [];

    /* Листы */
    protected $sheets;

    protected $line = 0;

    /**
     * PriceData constructor.
     * @param string $file
     * @param int|null $limitrows
     * @param string $encoding
     */
    public function __construct(string $file, int $limitrows = null, string $encoding = 'auto')
    {
        $this->file = $file;
        $this->limitrows = $limitrows;
        $this->encoding = $encoding;
    }

    public function getColCount(): int
    {
        return $this->colCount;
    }

    public function setColCount(int $colCount): void
    {
        $this->colCount = $colCount;
    }

    public function getLimitrows(): ?int
    {
        return $this->limitrows;
    }

    public function setLimitrows(int $limitrows = null): void
    {
        $this->limitrows = $limitrows;
    }

    /**
     * Добавление пачки записей
     * @param array $rows
     */
    public function addRows(array $rows): void
    {
        foreach ($rows as $row) {
            if ($this->getLimitrows() && $this->getCountRows() >= $this->getLimitrows()) {
                break;
            }

            if (!$row || !count($row)) {
                continue;
            }

            $this->addRow($row);

            unset($row);
        }
    }

    /**
     * Получение всех записей
     * @return array
     */
    public function getRows(): array
    {
        $rows = [];
        for ($i = 0; $i < $this->getCountRows(); $i++) {
            $rows[] = $this->getRow($i);
        }
        return $rows;
    }

    /**
     * @param int $i
     * @param array $row
     * @return bool
     */
    public function setData(int $i, array $row): bool
    {
        $this->data[$i] = gzcompress(serialize($row), 9);
        return true;
    }

    /**
     * @param int $i
     * @return array
     */
    public function getData(int $i): array
    {
        return unserialize(gzuncompress($this->data[$i]), ['allowed_classes' => false]);
    }

    /**
     * Добавление записи
     * @param array $row
     * @return bool
     */
    public function addRow(array $row): bool
    {
        if (is_array($row)) {
            $row = $this->convertToUtf($row);

            if ($this->setData($this->createKey($this->count_rows), $row)) {
                $this->count_rows++;
                if (max(array_keys($row)) + 1 > $this->getColCount()) {
                    $this->setColCount(max(array_keys($row)) + 1);
                }

                return true;
            }
        }

        $this->count_rows--;
        return false;
    }


    
    /**
     * Конвертация в системную кодировку
     * @param array $row
     * @return array
     */
    protected function convertToUtf(array $row): array
    {
        if ($this->notUseConvertToUtf === true) {
            $convert_to = $this->encoding;

            if ($this->encoding === 'auto' && !preg_match('/[а-яА-Я]/u', implode('', $row))) {
                $convert_to = 'windows-1251';
            }

            array_walk($row, static function (&$val_row) use ($convert_to) {
                $val_row = mb_convert_encoding($val_row, 'UTF-8', $convert_to);
            });

            return $row;
        }

        /** @noinspection PhpUnusedParameterInspection */
        $converter = static function (&$val_row, $key_row, $encoding = 'UTF-8') {
            ini_set('mbstring.substitute_character', 'none');
            $val_row = mb_convert_encoding($val_row, 'UTF-8', $encoding);
        };
        $encoding = !empty($this->encoding) ? $this->encoding : 'auto';
        $convert_to = false;

        //определяем кодировку данных если она отличается от системной то будем конвертировать
        if ($encoding !== 'auto' && $_SYSTEM->CHARSET !== $encoding) {
            $convert_to = $encoding;
        } elseif ($encoding === 'auto' && !preg_match('/[а-яА-Я]/u', implode('', $row))) {
            $convert_to = 'windows-1251';
        }

        //если необходма конвертация
        if ($convert_to) {
            array_walk($row, $converter, $convert_to);
        } else {
            array_walk($row, $converter);
        }

        return $row;
    }


    /**
     * уникальный ключ для конкретного файла
     * @param int $i
     * @return int
     */
    protected function createKey(int $i): int
    {
        return $i;
    }


    /**
     * получение записи по ключу
     * @param int $i
     * @return array
     */
    public function getRow(int $i): ?array
    {
        if ($i > $this->getCountRows()) {
            return null;
        }

        $row = $this->GetData($this->createKey($i));

        $this->line = 0;

        return $row;
    }


    /**
     * Общее кол-во записей
     * @return int
     */
    public function getCountRows(): int
    {
        return (int)$this->count_rows;
    }

    /**
     * Получение следующей записи
     */
    public function nextRow()
    {
        if (isset($this->data[$this->line]) && $row = $this->getData($this->line)) {
            $this->line++;
            return $row;
        }

        return false;
    }

    /**
     * @param array $rows
     * @return bool
     */
    public function setSheets(array $rows): bool
    {
        $this->sheets = gzcompress(serialize($rows), 9);
        return true;
    }

    /**
     * @return array
     */
    public function getSheets(): array
    {
        if (!$this->sheets) {
            return [0 => tr('По умолчанию')];
        }
        return unserialize(gzuncompress($this->sheets), ['allowed_classes' => false]);
    }
}
