<?php

namespace PriceReader;

use Exception;
use PriceReader\Exceptions\FileNotExist;
use RuntimeException;
use SplFileInfo;
use function unrar;

/**
 * Реализация контракта FileInterface
 *
 * @author Dmitriy
 */
class File extends SplFileInfo implements FileInterface
{

    /**
     * Путь к каталогу файла
     * @var string
     */
    protected $dir;

    /**
     * Наименование файла
     * @var string
     */
    protected $filename;

    protected $filepath;

    /**
     *
     * @var SplFileInfo $source
     */
    protected $source;


    /**
     * Расширение
     * @var string
     */
    protected $extension;

    /**
     * Constructs a new file from the given path.
     *
     * @param string $path The path to the file
     * @param bool $checkPath Whether to check the path or not
     *
     * @throws Exceptions\FileNotExist
     */
    public function __construct($path, $checkPath = true)
    {
        if ($checkPath && !is_file($path)) {
            throw new FileNotExist($path);
        }
        parent::__construct($path);
    }

    /**
     * транслитерация тексат с вырезанием левака
     * @param string $txt
     * @return string
     */
    public static function translit(string $txt): string
    {
        return strtolower(preg_replace('~[^A-Za-z0-9_\.]~', '', str_replace(' ', '_', translit($txt))));
    }


    /**
     * @return string
     */
    public function getFilepathDisplay(): string
    {
        if (!empty($this->filepath)) {
            return '/' . ltrim(str_replace($_SERVER['DOCUMENT_ROOT'], '', $this->filepath), '/');
        }

        return '';
    }

    public function isExist()
    {
        return is_file($this->getPathname());
    }

    
    /**
     * Заменяет обратные слэши (в windows) на прямые слэши и двойные слэши на одинарные.
     * @param string $path
     * @return string
     */
    protected function normalizePath($path): string
    {
        $path = str_replace('\\', '/', $path);
        return preg_replace('~\/{2,}~', '/', $path);
    }

    /**
     * перемещение файла
     * @param $directory
     * @param null $name
     * @return File
     */
    public function moveTo($directory, $name = null): File
    {
        $target = $this->getTargetFile($directory, $name);
        if (!@rename($this->getPathname(), $target)) {
            $error = error_get_last();
            throw new RuntimeException(sprintf(
                'Не могу переместить файл из "%s" в "%s" (%s)',
                $this->getPathname(),
                $target,
                strip_tags($error['message'])
            ));
        }
        @chmod($target, 0666 & ~umask());
        return $target;
    }
    
    protected function getTargetFile($directory, $name = null): self
    {
        if (!is_dir($directory)) {
            if (false === @mkdir($directory, 0777, true) && !is_dir($directory)) {
                throw new RuntimeException(sprintf('Не могу создать каталог "%s"', $directory));
            }
        } elseif (!is_writable($directory)) {
            throw new RuntimeException(sprintf('Не могу записать в каталог "%s"', $directory));
        }
        $target = rtrim($directory, '/\\') . DIRECTORY_SEPARATOR .
            (null === $name ? $this->getBasename() : $this->getName($name));
        return new self($target, false);
    }

    /**
     * Является ли файл архивом
     * @return boolean
     */
    public function isArchive(): bool
    {
        return in_array($this->getExtension(), ['zip', 'rar']);
    }

    /**
     * Извлечение файла из архива
     * @param null $concrete_file
     * @return bool
     * @throws Exception
     */
    public function unzipIfArchive($concrete_file = null): bool
    {
        if (!$this->isArchive()) {
            return false;
        }

        $dir = $this->getPath();

        $tmp = $dir . '/tmp';

        if ('zip' === $this->getExtension()) {
            unzip($this->getRealPath(), $tmp);
        } elseif ('rar' === $this->getExtension()) {
            unrar($this->getRealPath(), $tmp);
        } else {
            throw new RuntimeException(sprintf('Архив %s не поддерживается', $this->getExtension()));
        }

        unlink($this->getRealPath());

        $dh = opendir($tmp);

        if (!$dh) {
            throw new RuntimeException('Временная директория для распаковки файлов архива не может быть создана!');
        }

        $new = $this->getRealPath();
        $cnt = 0;

        //проход по архиву и копирование Первого файла от туда
        //или конкретного если есть $concrete_file
        while (($file = readdir($dh)) !== false) {
            $old = $tmp . '/' . $file;

            if (in_array($file, array('.', '..')) || !is_file($old)) {
                continue;
            }

            if (!empty($concrete_file) && stripos($file, $concrete_file) === false) {
                continue;
            }

            $cnt++;
            $new = $dir . '/' . translit_file($file);

            if (file_exists($new) && hash_file('crc32', $new) === hash_file('crc32', $old)) {
                closedir($dh);
                rmpath2($tmp);
                /**
                 * @todo Перевести на нормальный FileExistexception с передачей файла
                 */
                throw new RuntimeException(
                    'Аналогичный файл уже существует, нет необходимости обновлять!',
                    new File($new)
                );
            }

            \copy_convert_utf($old, $new);
            break;
        }
        closedir($dh);

        rmpath2($tmp);

        if ($cnt === 0) {
            throw new RuntimeException('Файлы в архиве отсутсвуют!');
        }

        $this->__construct($new);

        return true;
    }

    /**
     * Удаление текущего файла
     */
    public function delete()
    {
        return $this->isExist() && @unlink($this->getRealPath());
    }

    /**
     * Получить хеш содержимого
     * @param string $mode
     * @return string
     */
    public function getContentHash($mode = 'crc32'): string
    {
        return hash_file($mode, $this->getRealPath());
    }

    /**
     * Получить содержимое файла
     */
    public function getContent()
    {
        return file_get_contents($this->getRealPath());
    }

    /**
         * Returns locale independent base name of the given path.
         * @param string $name The new file name
         *
         * @return string containing
         */
    protected function getName($name): string
    {
        $originalName = str_replace('\\', '/', $name);
        $pos = strrpos($originalName, '/');
        $originalName = false === $pos ? $originalName : substr($originalName, $pos + 1);
        return $originalName;
    }
}
