<?php

namespace PriceReader;

use PriceReader\Data\PriceData;
use PriceReader\Exceptions\ParceError;
use PriceReader\Exceptions\ExtNotSupport;
use PriceReader\Exceptions\FileNotExist;
use PriceReader\Readers\CsvReader;
use PriceReader\Readers\XlsReader;
use PriceReader\Readers\XlsxReader;
use RuntimeException;
use XlsParser\OLEIncorrectException;

/**
 * Основной фасадный класс
 *
 * @author d.lanec
 */
class Reader
{

    /**
     * @var array
     */
    protected static $support_ext = ['csv', 'xls','xlsx', 'txt'];

    /**
     * Объект файла
     * @var \SplFileInfo $SplFile
     */
    protected $SplFile;

    /**
     * предпологаемая кодировка файла
     * @var string
     */
    protected $encoding;

    /**
     * временный каталог необходимый для работы с xlsx
     * @var string
     */
    protected $tmpdir;

    /**
     * Принудительный разделитель csv
     *
     * @var null|string
     */
    protected $csvDelimeter;

    /**
     * Читалка файлов
     * @param File $SplFile
     * @param string $tmpdir временный каталог, абсолютный путь
     * @param string $encoding кодировка файла
     * @param int $fileSizeLimit лимит размера в Мб
     * @throws ExtNotSupport
     * @throws FileNotExist
     */
    public function __construct(
        File $SplFile,
        string $tmpdir,
        string $encoding = 'auto',
        int $fileSizeLimit = null
    ) {
        if (!in_array(strtolower($SplFile->getExtension()), static::$support_ext, true)) {
            throw new ExtNotSupport($SplFile->getExtension());
        }

        if (!$SplFile->isExist()) {
            throw new FileNotExist($SplFile->getRealPath());
        }

        //ограничение по размеру
        if ($fileSizeLimit) {
            $size = round($SplFile->getSize() / 1024 / 1024);
            if ($size > $fileSizeLimit) {
                throw new RuntimeException(sprintf(
                    'Размер файла %sMb превышает предельный размер %sMb, чтение файла не возможно',
                    $size,
                    $fileSizeLimit
                ));
            }
        }

        $this->SplFile      = $SplFile;
        $this->encoding     = !empty($encoding) ? $encoding : 'auto';
        $this->tmpdir       = $tmpdir;
    }

    /**
     * поддержимаемые расширения
     * @return array
     */
    public function getSupportExt(): array
    {
        return static::$support_ext;
    }

    /**
     * @return null|string
     */
    public function getCsvDelimeter(): ?string
    {
        return $this->csvDelimeter;
    }

    /**
     * @param null|string $csvDelimeter
     * @return Reader
     */
    public function setCsvDelimeter($csvDelimeter): self
    {
        $this->csvDelimeter = $csvDelimeter;

        return $this;
    }

    /**
     * загрузка с пользовательской функцией
     *
     * @param PriceData $PriceData
     * @param callable  $handler
     *
     * @return bool
     */
    public function walkPriceData(PriceData $PriceData, callable $handler): bool
    {
        if ($PriceData && $PriceData->getCountRows()> 0) {
            $i = 0;

            while ($row = $PriceData->nextRow()) {
                $i++;

                if (!$handler($row, $PriceData->getCountRows(), $i)) {
                    break;
                }
            }
        }

        return true;
    }

    /**
     * Получение данных из файла
     *
     * @param int $limitrows
     * @param int $sheet
     * @return PriceData
     * @throws ExtNotSupport
     */
    public function getPriceData(int $limitrows = null, int $sheet = 0): PriceData
    {
        switch (strtolower($this->SplFile->getExtension())) {
            case 'csv':
            case 'txt':
                $CsvReader = new CsvReader($this->SplFile, $this->encoding, $this->csvDelimeter);
                if ($this->csvDelimeter) {
                    $CsvReader->setDelimeter($this->csvDelimeter);
                }
                $PriceData = $CsvReader->getPriceData($limitrows, $sheet);
                break;
            case 'xls':
                try {
                    $PriceData = (new XlsReader($this->SplFile, $this->encoding))->getPriceData($limitrows, $sheet);
                } catch (OLEIncorrectException $exc) {
                    try {
                        $PriceData = (new XlsxReader($this->SplFile, $this->encoding, $this->tmpdir))
                            ->getPriceData($limitrows, $sheet);
                    } catch (ParceError $e) {
                        $CsvReader = new CsvReader($this->SplFile, $this->encoding, $this->csvDelimeter);
                        if ($this->csvDelimeter) {
                            $CsvReader->setDelimeter($this->csvDelimeter);
                        }
                        $PriceData = $CsvReader->getPriceData($limitrows, $sheet);
                    }
                }

                break;
            case 'xlsx':
                try {
                    $PriceData = (new XlsxReader($this->SplFile, $this->encoding, $this->tmpdir))
                        ->getPriceData($limitrows, $sheet);
                } catch (ParceError $e) {
                    try {
                        $PriceData = (new Readers\XlsReader($this->SplFile, $this->encoding))
                            ->getPriceData($limitrows, $sheet);
                    } catch (OLEIncorrectException $exc) {
                        $CsvReader = new Readers\CsvReader($this->SplFile, $this->encoding, $this->csvDelimeter);
                        if ($this->csvDelimeter) {
                            $CsvReader->setDelimeter($this->csvDelimeter);
                        }
                        $PriceData = $CsvReader->getPriceData($limitrows, $sheet);
                    }
                }
                break;

            default:
                throw new ExtNotSupport($this->SplFile->getExtension());
        }

        return $PriceData;
    }
}
