<?php

namespace PriceReader\Data;

use RuntimeException;

/**
 * Description of LazyPriceData
 *
 * @author d.lanec
 */
class CsvPriceData extends PriceData
{
    protected $source;

    protected $delimiter;

    protected $line = 0;

    protected $columns;

    public static $LENGHT_LINE = 81920;

    /**
     * CsvPriceData constructor.
     * @param string $filename
     * @param int|null $limitrows
     * @param string $encoding
     */
    public function __construct(string $filename, int $limitrows = null, string $encoding = 'auto')
    {
        parent::__construct($filename, $limitrows, $encoding);

        $this->setSource(fopen($filename, 'rb'));

        if (!$delimiter = $this->definitionDelimiter($filename)) {
            throw new RuntimeException('Ошибка при чтении файла, разделитель не обнаружен.');
        }

        $this->setDelimiter($delimiter);

        $this->calculateCountRows();
    }

    /**
     * Получение всех данных
     * @return array
     */
    public function getRows(): array
    {
        $rows = [];
        while ($row = $this->nextRow()) {
            $rows[] = $row;
        }

        return $rows;
    }

    public function getDelimiter(): ?string
    {
        return $this->delimiter;
    }

    public function setDelimiter(string $delimiter): void
    {
        $this->delimiter = $delimiter;
        $this->calculateColCount();
    }

    public function getSource()
    {
        return $this->source;
    }

    public function setSource($source): void
    {
        $this->source = $source;
    }

    public function getLine(): int
    {
        return $this->line;
    }

    public function nextRow()
    {
        if ($this->limitrows && $this->line >= $this->limitrows) {
            return false;
        }

        if (!$line = fgets($this->getSource(), self::$LENGHT_LINE)) {
            return false;
        }

        $row = str_getcsv($line, $this->getDelimiter());
        $this->line++;

        $row = $this->convertToUtf($row);

        return $row;
    }

    /**
     * автоопределение разделителя
     * @param string $filename
     * @return string|bool
     */
    public function definitionDelimiter($filename)
    {
        $checkLines = 20;

        $file = new \SplFileObject($filename);
        $delimiters = array(';', "\t", ',', '|', ':');
        $results = [];
        $i = 0;
        while ($file->valid() && $i <= $checkLines) {
            $line = $file->fgets();
            foreach ($delimiters as $delimiter) {
                $regExp = '/[' . $delimiter . ']/';

                if (strlen($line) - strlen(preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $line)) > 2) {
                    continue;
                }
                $fields = preg_split($regExp, $line);
                if (count($fields) > 1) {
                    if (!empty($results[$delimiter])) {
                        $results[$delimiter]++;
                    } else {
                        $results[$delimiter] = 1;
                    }
                }
            }
            $i++;
        }

        if (empty($results)) {
            return false;
        }

        //начисляем очки с учетом приоритера нахождения разделителя в массиве $delimiters,
        //при условии что есть равные по очкам разделители
        foreach ($results as $del => $score) {
            $results[$del] += (count($delimiters) - array_search($del, $delimiters, false));
        }

        $results = array_keys($results, max($results));

        return $results[0];
    }

    /**
     * Общее кол-во записей
     */
    public function calculateCountRows(): void
    {
        fseek($this->getSource(), 0);

        while ($line = fgets($this->getSource(), self::$LENGHT_LINE)) {
            $this->count_rows++;

            if ($this->limitrows && $this->count_rows >= $this->limitrows) {
                break;
            }
        }

        fseek($this->getSource(), 0);
    }

    public function calculateColCount(): void
    {
        fseek($this->getSource(), 0);

        $i = 0;

        while ($line = fgets($this->getSource(), self::$LENGHT_LINE)) {
            $i++;

            $row = str_getcsv($line, $this->getDelimiter());
            if (count($row) > $this->getColCount()) {
                $this->setColCount(count($row));
            }

            if ($i > 20) {
                break;
            }
        }

        fseek($this->getSource(), 0);
    }


    public function __destruct()
    {
        if ($this->getSource()) {
            fclose($this->getSource());
        }
    }

    /**
     * получение записи по ключу
     *
     * @param int $i
     *
     * @return null|array
     */
    public function getRow(int $i): ?array
    {
        if ($i > $this->getCountRows()) {
            return null;
        }

        $t = 0;

        while (($line = fgets($this->getSource(), self::$LENGHT_LINE)) && $t < $i) {
            $t++;
        }

        fseek($this->getSource(), 0);

        return $this->convertToUtf(str_getcsv($line, $this->getDelimiter()));
    }
}
