<?php

namespace Shop\Presentation\Modules\AdminPriceListsAccess;

use ActiveTable\ColumnTable;
use Core\Form\ControlBuilder;
use Diamond\Helpers\Form;
use DiamondTable\TableFactory;
use Shop\Domain\PriceAccess\Contracts\PriceAccessEntityInterface;
use Shop\Infrastructure\Repositories\Customer\CustomerRepository;
use Shop\Infrastructure\Repositories\CustomerGroup\CustomerGroupCriteria;
use Shop\Infrastructure\Repositories\CustomerGroup\CustomerGroupRepository;
use Shop\Infrastructure\Repositories\PriceAccess\PriceAccessCriteria;
use Shop\Infrastructure\Repositories\PriceAccess\PriceAccessRepository;
use Shop\Infrastructure\Repositories\PriceList\PriceListCriteria;
use Shop\Infrastructure\Repositories\PriceList\PriceListRepository;
use Shop\Infrastructure\Repositories\Stock\StockCriteria;
use Shop\Infrastructure\Repositories\Stock\StockRepository;

/**
 * Description of AdminPriceListsAccess
 *
 * @author Diamond Code Generator
 */
class AdminPriceListsAccess extends \Core\ModuleController
{


    public function __construct(protected PriceAccessRepository $priceAccessRepository,
                                protected  TableFactory $tableFactory,
                                protected ControlBuilder $controlBuilder,
                                protected CustomerRepository $customerRepository,
                                protected StockRepository $stockRepository,
                                protected CustomerGroupRepository $customerGroupRepository,
                                protected PriceListRepository $priceListRepository)
    {

    }

    public function process(): string
    {
        $table = $this->tableFactory->buildAdminTable($this->priceAccessRepository, PriceAccessCriteria::create());

        $table
            ->addColumn(
                (new ColumnTable('id', '№'))->setWidth(40)
            )
            ->addColumn(
                (new ColumnTable('customerGroupId', 'Группа клиента'))
                    ->setFormat($this, 'formatCustomerGroup')
            )
            ->addColumn(
                (new ColumnTable('customerId', 'Клиент'))
                    ->setFormat($this, 'formatCustomer')
            )
            ->addColumn(
                (new ColumnTable('shopId', 'Торговая точка'))
                    ->setFormat($this, 'formatStock')
            )
            ->addColumn(
                (new ColumnTable('priceScheetId', 'Прайс-лист'))
                    ->setFormat($this, 'formatPriceList')
            )
            ->addColumn(
                (new ColumnTable('actions', ''))
                    ->setFormat($this->tableFactory, 'formatActions')
                    ->setWidth(120)
            );

        $groups = $this->customerGroupRepository->findByCriteria(
            CustomerGroupCriteria::create()
        );

        $prices = $this->priceListRepository->findByCriteria(
            PriceListCriteria::create()
        );

        $stocks = $this->stockRepository->findByCriteria(
            StockCriteria::create()->setFilterByActive(true)->setSortByTitle('asc')
        );

        $caption = 'любой клиент';

        if (input_get('id')) {
            $access = $this->priceAccessRepository->findById(input_get('id'));

            if ($access->getCustomerId() > 0) {
                $customer = $this->customerRepository->findById($access->getCustomerId());
                $caption = $customer->getName();
            }

        }

        $table
            ->addField(
                $this->controlBuilder->buildAutoDropdown(
                    'priceScheetId',
                    Form::prepareDropdownWithDef($prices, '%','любой прайс-лист', 'getDropdownListWithFileName')
                )
            )
            ->addField(
                $this->controlBuilder->buildDropdownUniform('customerGroupId',
                    Form::prepareDropdownWithDef($groups,'%','любая группа клиента'))
            )
            ->addField(
                $this->controlBuilder->buildAutoDropdown(
                    'shopId',
                    Form::prepareDropdownWithDef($stocks, '%','любая торговая точка')
                )
            )
            ->addField(
                $this->controlBuilder->buildWindowSelect(
                    'customerId',
                    '/admin/shop/customers/customers-lite',
                    $caption
                )->setValue('%')
            );

        return $table->render();
    }

    /**
     * @param PriceAccessEntityInterface $accessEntity
     * @return string
     */
    public function formatCustomerGroup(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getCustomerGroupId() > 0) {
            $group = $this->customerGroupRepository->findById($accessEntity->getCustomerGroupId());
            return '#'.$group->getId().' '.$group->getName();
        }
        return 'любая';
    }

    public function formatPriceList(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getPriceScheetId() > 0) {
            $price = $this->priceListRepository->findById($accessEntity->getPriceScheetId());
            return '#'.$price->getId().' '.$price->getTitle();
        }
        return 'любой';
    }

    /**
     * @param PriceAccessEntityInterface $accessEntity
     * @return string
     */
    public function formatCustomer(PriceAccessEntityInterface $accessEntity): string
    {

        if ($accessEntity->getCustomerId() > 0) {
            $customer = $this->customerRepository->findById($accessEntity->getCustomerId());
            if($customer){
                return anchor('/admin/shop/customers/customers?filter',$customer->getName(), [], ['filter_by_id' => $customer->getId()]);
            }
            else{
                return '#'.$accessEntity->getCustomerId();
            }

        }
        return 'любой';
    }

    public function formatStock(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getShopId() > 0) {
            $shop = $this->stockRepository->findById($accessEntity->getShopId());
            return '#'.$shop->getId().' '.$shop->getTitle();
        }
        return 'любая';
    }

}               