<?php

namespace Shop\Presentation\Modules\AdminPriceLists;

use ActiveTable\ColumnTable;
use Core\Form\ControlBuilder;
use Core\Helpers\Date;
use Core\ModuleController;
use Diamond\Helpers\Form;
use DiamondTable\TableFactory;
use Shop\Domain\PriceList\Contracts\PriceListEntityInterface;
use Shop\Infrastructure\Repositories\Currency\CurrencyCriteria;
use Shop\Infrastructure\Repositories\Currency\CurrencyRepository;
use Shop\Infrastructure\Repositories\PriceFile\PriceFileCriteria;
use Shop\Infrastructure\Repositories\PriceFile\PriceFileRepository;
use Shop\Infrastructure\Repositories\PriceList\PriceListCriteria;
use Shop\Infrastructure\Repositories\PriceList\PriceListRepository;
use Shop\Infrastructure\Repositories\Provider\ProviderCriteria;
use Shop\Infrastructure\Repositories\Provider\ProviderRepository;

/**
 * Description of AdminPriceLists
 *
 * @author Diamond Code Generator
 */
class AdminPriceLists extends ModuleController
{

    protected $priceListRepository;
    protected $tableFactory;
    protected $controlBuilder;
    protected $priceFileRepository;
    protected $currencyRepository;
    protected $providerRepository;

    /**
     * AdminPriceLists constructor.
     * @param PriceListRepository $priceListRepository
     * @param TableFactory $tableFactory
     * @param ControlBuilder $controlBuilder
     * @param PriceFileRepository $priceFileRepository
     * @param CurrencyRepository $currencyRepository
     * @param ProviderRepository $providerRepository
     */
    public function __construct(PriceListRepository $priceListRepository, TableFactory $tableFactory,
                                ControlBuilder $controlBuilder,
                                PriceFileRepository $priceFileRepository, CurrencyRepository $currencyRepository,
                                ProviderRepository $providerRepository)
    {
        $this->priceListRepository = $priceListRepository;
        $this->tableFactory = $tableFactory;
        $this->controlBuilder = $controlBuilder;
        $this->priceFileRepository = $priceFileRepository;
        $this->currencyRepository = $currencyRepository;
        $this->providerRepository = $providerRepository;
    }


    public function process(): string
    {

        $table = $this->tableFactory->buildAdminTable($this->priceListRepository, PriceListCriteria::create());

        $providers = $this->providerRepository->findByCriteria(
            ProviderCriteria::create()->setSortByName('ASC')
        );


        $pricefiles = $this->priceFileRepository->findByCriteria(
            PriceFileCriteria::create()->setSortByName('ASC')
        );

        $currencies = $this->currencyRepository->findByCriteria(
            CurrencyCriteria::create()
        );

        $providersAr = Form::prepareDropdown($providers);
        $pricefilesAr = Form::prepareDropdown($pricefiles);
        $currenciesAr = Form::prepareDropdown($currencies);

        $table
            ->setDefaultSortColumn('id', 'DESC')
            ->addColumn(
                (new ColumnTable('id', '№'))->setWidth(40)
            )
            ->addColumn(
                (new ColumnTable('title', 'Название прайс-листа'))
            )
            ->addColumn(
                (new ColumnTable('count', 'Кол-во позиций'))
                    ->setFormat($this, 'formatCount')
            )
            ->addColumn(
                (new ColumnTable('dateUpdate', 'Дата обновления'))
                    ->setFormat($this, 'formatDate')
                    ->setSorted(true)
            )
            ->addColumn(
                (new ColumnTable('pricelistFileId', 'Прайс-файл'))
                    ->setFormat($this, 'formatPriceListFile')
            )
            ->addColumn(
                (new ColumnTable('providerId', 'Поставщик'))
                    ->setFormat($this, 'formatProvider')
            )
            ->addColumn(
                (new ColumnTable('currencyId', 'Валюта прайса'))
                    ->setFormat($this, 'formatCurrency')
            )
            ->addColumn(
                (new ColumnTable('actions', ''))
                    ->setFormat($this->tableFactory, 'formatActions')
                    ->setWidth(120)
            );

        if (input_get('id')) {
            $table
                ->addField(
                    $this->controlBuilder->buildTextLabel('count')->setFn($this, 'formatCount')
                )
                ->addField(
                    $this->controlBuilder->buildTextLabel('dateUpdate')->setFn($this, 'formatDate')
                );

        }

        $table
            ->addField(
                $this->controlBuilder->buildInput('title'), true
            )
            ->addField(
                $this->controlBuilder->buildDropdownUniform('pricelistFileId', $pricefilesAr), true
            )
            ->addField(
                $this->controlBuilder->buildDropdownUniform('currencyId', $currenciesAr), true
            );


        $table
            ->addFilter(
                $this->controlBuilder->buildDropdownUniform('filter_by_providerId', $providersAr)
            )
            ->addFilter(
                $this->controlBuilder->buildDropdownUniform('filter_by_pricelistFileId', $pricefilesAr)
            )
            ->addFilter(
                $this->controlBuilder->buildInput('search_by_title')
            )
        ;

        return $table->render();
    }

    /**
     * @param PriceListEntityInterface $price
     * @return string
     */
    public function formatCurrency(PriceListEntityInterface $price): string
    {
        return $price->getCurrency()->getHtmlSign();
    }

    /**
     * @param PriceListEntityInterface $priceListSheet
     * @return string
     */
    public function formatCount(PriceListEntityInterface $priceListSheet): string
    {
        $c = $priceListSheet->getPositions()->count();
        return (string)(int)$c;
    }

    /**
     * @param PriceListEntityInterface $priceListSheet
     * @return string
     */
    public function formatDate(PriceListEntityInterface $priceListSheet): string
    {
        if (!$priceListSheet->getDateUpdate()) {
            return '';
        }
        return '<nobr>' . $priceListSheet->getDateUpdate(Date::formatDisplay()) . '</nobr>';
    }

    /**
     * @param PriceListEntityInterface $priceListSheet
     * @return string
     */
    public function formatPriceListFile(PriceListEntityInterface $priceListSheet): string
    {
        $name = $priceListSheet->getPricelistFile()->getName();
        return anchor('/admin/shop/catalog/price-files',$name,[],[
            'fn'    => 'edit',
            'id'    => $priceListSheet->getPricelistFile()->getId()
        ]);
    }

    /**
     * @param PriceListEntityInterface $priceListSheet
     * @return string
     */
    public function formatProvider(PriceListEntityInterface $priceListSheet): string
    {
        $name = $priceListSheet->getPricelistFile()->getProvider()->getName();
        return anchor('/admin/shop/stocks/providers',$name,[],[
            'fn'    => 'edit',
            'id'    => $priceListSheet->getPricelistFile()->getProvider()->getId()
        ]);
    }
}