<?php

namespace Shop\Application\Catalog;

use Diamond\Application\Imaging\ImagingService;
use Shop\Application\Catalog\ProductNotFoundException;
use Shop\Infrastructure\Models\Product\Product;
use Shop\Presentation\Config\Main;
use Shop\Domain\Customer\Contracts\CustomerEntityInterface;
use Shop\Domain\Product\Contracts\ProductEntityInterface;
use Shop\Domain\Product\Contracts\ProductThumbCreaterInterface;
use Shop\Infrastructure\Repositories\Product\ProductCriteria;
use Shop\Infrastructure\Repositories\Product\ProductRepository;
use Shop\Presentation\Helpers\ProductHelper;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * Class CatalogService
 * @package Shop\Application\Catalog
 */
class CatalogService implements ProductThumbCreaterInterface
{


    protected $productRepository;
    protected $mainConfig;

    protected $session;
    protected $imagingService;

    /**
     * CatalogService constructor.
     * @param ProductRepository $productRepository
     * @param Session $session
     * @param ImagingService $imagingService
     * @param Main $mainConfig
     */
    public function __construct(ProductRepository $productRepository,
                                Session $session,
                                ImagingService $imagingService,
                                Main $mainConfig)
    {
        $this->productRepository = $productRepository;
        $this->mainConfig = $mainConfig;
        $this->session = $session;
        $this->imagingService = $imagingService;
    }


    /**
     * @return int
     */
    public function getProductsCount(): int
    {
        return $this->productRepository->count(ProductCriteria::create());
    }




    /**
     * путь к каталогу изображений
     * @return string
     */
    public function getProductsImageDir(): string
    {
        return $this->imagingService->getUploadDir() . $this->mainConfig->getProductImagePath();
    }


    /**
     * @param string $path
     * @param int $productId
     * @param string|null $getImg
     * @return string
     */
    public function createThumbPath(string $path, int $productId, ?string $getImg): string
    {
        if (!$getImg) {
            return '/assets/images/no-photo.png';
        }
        return sprintf('%s%s_%s', $path, str_pad(
            $productId, 8, 0, STR_PAD_LEFT
        ), $getImg);
    }


    /**
     * @param int $id
     * @return ProductEntityInterface
     * @throws \Repo\Concrete\Exceptions\Collection
     */
    public function getProductById(int $id): ProductEntityInterface
    {
        /**
         * @var ProductEntityInterface $product
         */
        $criteria = ProductCriteria::create()
            ->setNewGoodDays($this->mainConfig->getDaysNewGoods())
            ->setLimit(1)
            ->setFilterById($id)
        ;
        $products = $this->productRepository->findByCriteria($criteria);

        if (!$product = $products->current()) {
            throw new ProductNotFoundException('product not found by id ' . $id);
        }

        $ids = (array)$this->session->get('favorites');

        if(in_array($product->getId(), $ids, false)){
            $product->setFavorite();
        }

        $ids2 = (array)$this->session->get('export');

        if(in_array($product->getId(), $ids2, false)){
            $product->setExport();
        }

        return $product;
    }


    /**
     * @param ProductEntityInterface $product
     * @param CustomerEntityInterface $customer
     * @return array
     */
    public function getProductImages(ProductEntityInterface $product, CustomerEntityInterface $customer): array
    {
        $images = [];

        $server = $this->mainConfig->getImagesServer();
        $thumbPath = $server . $this->mainConfig->getThumbPath();
        $fullPath = $server . '/image/';

        foreach (Product::$images as $image) {

            $img = $product->{'get'.$image}();


            if(empty($img)){
                continue;
            }

            $image_full_path = ProductHelper::getImagePath($this->mainConfig , $product->{'get'.$image}()) ;
            $thumb_path = ProductHelper::getThumbImagePath($this->mainConfig , $product->{'get'.$image}()) ;

            $images[md5($image)] = [
                'thumb' => $thumb_path,
                'full' => $image_full_path
            ];
        }
        
        return $images;
    }
}