<?php


namespace Shop\Application\Shopping\SendCustomerOrders;


use Diamond\Application\Notification\Exceptions\NotificationException;
use Diamond\Application\Notification\NotificationService;
use Shop\Application\Office\OfficeService;
use Shop\Application\Shopping\PrepareCustomerOrderDataForReport\PrepareCustomerOrderDataForReportCommand;
use Shop\Application\Shopping\PrepareCustomerOrderDataForReport\ResultData;
use Shop\Application\Shopping\ShoppingService;
use Shop\Domain\CustomerOrder\CustomerOrderCollection;
use Shop\Infrastructure\Repositories\CustomerOrder\CustomerOrderRepository;

class SendCustomerOrdersHandler
{
    protected $customerOrderRepository;
    protected $notificationService;
    protected $shoppingService;

    /**
     * SendCustomerOrdersHandler constructor.
     * @param CustomerOrderRepository $customerOrderRepository
     * @param ShoppingService $shoppingService
     * @param NotificationService $notificationService
     */
    public function __construct(CustomerOrderRepository $customerOrderRepository,
                                ShoppingService $shoppingService,
                                NotificationService $notificationService)
    {
        $this->customerOrderRepository = $customerOrderRepository;
        $this->notificationService = $notificationService;
        $this->shoppingService = $shoppingService;
    }

    /**
     * @param SendCustomerOrdersCommand $command
     * @throws \Repo\Concrete\Exceptions\Collection
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function handle(SendCustomerOrdersCommand $command): void
    {
        $errors = [];

        $successOrders = new CustomerOrderCollection();

        $ids = $command->getIds();

        $notifyCustomer = $command->isNotifyCustomer();
        $notifyManager = $command->isNotifyManager();

        /**
         * @var Order $order
         */
        foreach ($ids as $id) {

            if (!$order = $this->customerOrderRepository->findById($id)) {
                $errors[] = $id;
            }

            $data = $this->shoppingService->prepareCustomerOrderDataForReport($order);

            try {

                if ($notifyCustomer === true) {
                    $email = $order->getCustomer()->getEmail();
                    $this->notificationService->sendToEmailByReport('order_notify', $email, $data);
                }

                if ($notifyManager === true) {
                    $email = $order->getCustomer()->getManager()->getEmail();
                    $this->notificationService->sendToEmailByReport('order_notify_manager', $email, $data);
                }


                $successOrders->push($order);

            } catch (NotificationException $ex) {
                $errors[] = $order->getId();
            }
        }


        if (count($errors)) {
            throw new SendCustomerOrdersException($successOrders, $errors, 'send customer orders errors: ' . implode(',', $errors));
        }



        $command->getResultOrders()->setSuccessOrders($successOrders);
    }

}