<?php


namespace Shop\Infrastructure\Repositories\Brand;


use Diamond\Repositories\PropelAbstractRepository;
use Propel\Runtime\ActiveQuery\Criteria;
use Propel\Runtime\ActiveQuery\ModelCriteria;
use Repo\CollectionInterface;
use Repo\Concrete\AbstractCollection;
use Repo\EntityInterface;
use Repo\PaginationInterface;
use Shop\Contracts\CatalogFilterInterface;
use Shop\Domain\Brand\BrandCollection;
use Shop\Domain\Brand\Contracts\BrandEntityInterface;
use Shop\Domain\Brand\Contracts\BrandRepositoryInterface;
use Shop\Domain\Customer\Contracts\CustomerEntityInterface;
use Shop\Domain\Product\Contracts\ProductCriteriaInterface;
use Shop\Infrastructure\Models\Brand\Brand;
use Shop\Infrastructure\Models\Brand\BrandQuery;

class BrandRepository extends PropelAbstractRepository implements BrandRepositoryInterface
{
    /**
     * @return EntityInterface
     */
    public static function createEntity(): EntityInterface
    {
        return new Brand();
    }

    protected function createQuery(): ModelCriteria
    {
        return BrandQuery::create();
    }

    /**
     * @param PaginationInterface|BrandCriteria $criteria
     * @param ModelCriteria|BrandQuery $dbCriteria
     */
    protected function modifyCriteria(PaginationInterface $criteria, ModelCriteria $dbCriteria): void
    {

        $dbCriteria
            ->_if($criteria->getFilterByName() !== null)
            ->filterByName($criteria->getFilterByName())
            ->_endif()
            ->_if($criteria->getSearchByName() !== null)
            ->filterByName('%' . $criteria->getSearchByName() . '%', Criteria::LIKE)
            ->_endif()
            ->_if($criteria->getSearchByAlias() !== null)
            ->filterByAlias('%' . $criteria->getSearchByAlias() . '%', Criteria::LIKE)
            ->_endif()
            ->_if($criteria->getFilterByAlias() !== null)
            ->filterByAlias($criteria->getFilterByAlias())
            ->_endif()
            //------------------------
            ->_if($criteria->getFilterByDestinations() !== null)
            ->useProductQuery()
            ->usePositionQuery()
            ->usePricelistSheetQuery()
            ->usePricelistFileQuery()
            ->useProviderQuery()
            ->filterByDestination($criteria->getFilterByDestinations())
            ->endUse()
            ->endUse()
            ->endUse()
            ->endUse()
            ->endUse()
            ->_endif()
            //------------------------
            ->_if($criteria->getFilterByProductCategoryId() !== null)
            ->useProductQuery()
            ->filterByCategoryId($criteria->getFilterByProductCategoryId())
            ->endUse()
            ->_endif()
            ->_if($criteria->getSearchByProductName() !== null)
            ->useProductQuery()
            ->filterByTitle('%' . $criteria->getSearchByProductName() . '%', Criteria::LIKE)
            ->endUse()
            ->_endif()
            ->_if(is_array($criteria->getFilterByWeights()) && count($criteria->getFilterByWeights()))
            ->useProductQuery()
            ->filterByWeight($criteria->getFilterByWeights())
            ->endUse()
            ->_endif()
            ->_if($criteria->getFilterByPriceListScheetIds() !== null)
            ->useProductQuery()
            ->usePositionQuery()
            ->filterByPricelistSheetId($criteria->getFilterByPriceListScheetIds())
            ->endUse()
            ->endUse()
            ->_endif()
            ->_if($criteria->getFilterByAlias() !== null)
            ->filterByAlias($criteria->getFilterByAlias())
            ->_endif()
            ->_if($criteria->getSortByName() !== null)
            ->orderByName($criteria->getSortByName())
            ->_endif()
            ->groupById();
    }



    protected function createCollection(): CollectionInterface
    {
        return new BrandCollection();
    }


    /**
     * @param int $shopId
     * @param array $ignore
     * @param array $orderAlias
     * @param $limit
     * @return BrandCollection
     */
    public function findWithImagesByShopIdWithoutHideBrands(int $shopId, array $ignore, array $orderAlias = [],
                                                                $limit = false): BrandCollection
    {

        $results = $this
            ->createQuery()
            ->useProductQuery()
            ->filterByImg('', Criteria::GREATER_THAN)
            ->usePositionQuery()
            ->filterByRemain(0, Criteria::GREATER_THAN)
            ->usePricelistSheetQuery()
            ->usePricelistFileQuery()
            ->useProviderQuery()
            ->filterByShopId($shopId)
            ->endUse()
            ->endUse()
            ->endUse()
            ->endUse()
            ->endUse()
            ->filterById($ignore, Criteria::NOT_IN)
            ->filterByAlias('', Criteria::GREATER_THAN)
            ->groupById()
            ->_if(count($orderAlias))
            ->addDescendingOrderByColumn(sprintf(
                "FIELD(`shop_brands`.`alias` ,'%s')", implode("','", $orderAlias)
            ))
            ->_endif()
            ->orderByAlias()
            ->addAscendingOrderByColumn('rand()')
            ->_if($limit)
            ->limit($limit)
            ->_endif()
            ->find();

        $collection = new BrandCollection();
        foreach ($results as $row) {
            $collection->push($row);
        }

        return $collection;
    }

    /**
     * @param ProductCriteriaInterface $catalogFilter
     * @return BrandCollection
     * @throws \Repo\Concrete\Exceptions\Collection
     */
    public function findWithImagesByCustomerAccess(ProductCriteriaInterface $catalogFilter): BrandCollection
    {

        $results = $this
            ->createQuery()
            ->useProductQuery()
            ->filterByImg('', Criteria::GREATER_THAN)
            ->usePositionQuery()
            ->filterByRemain(0, Criteria::GREATER_THAN)
            ->filterByImage('', Criteria::GREATER_THAN)
            ->_if($catalogFilter->getFilterByPriceListScheetIds() !== null)
            ->filterByPricelistSheetId($catalogFilter->getFilterByPriceListScheetIds())
            ->_endif()
            ->_if($catalogFilter->getFilterBySaleId() !== null)
            ->filterBySaleId($catalogFilter->getFilterBySaleId())
            ->_endif()
            ->endUse()
            ->endUse()
            ->groupById()
            ->orderByName()
            ->find();

        $collection = new BrandCollection();
        foreach ($results as $row) {
            $collection->push($row);
        }

        return $collection;
    }


    /**
     *
     * @param type $title
     * @return type
     */
    public function findByName($title, CatalogFilterInterface $catalogFilter): BrandCollection
    {
        $results = $this
            ->createQuery()
            ->filterByName("{$title}%", Criteria::LIKE)
            ->orderByName()
            ->useProductQuery()
            ->usePositionQuery()
            ->filterByRemain(0, Criteria::GREATER_THAN)
            ->filterByPricelistSheetId($catalogFilter->getPriceListScheetIds())
            ->endUse()
            ->filterByImg('', Criteria::GREATER_THAN)
            ->endUse()
            ->limit(10)
            ->groupById()
            ->find();

        $collection = new BrandCollection();
        foreach ($results as $row) {
            $collection->push($row);
        }

        return $collection;
    }

    /**
     * @param string $alias
     * @return BrandEntityInterface|null
     */
    public function findOneByAlias(string $alias): ?BrandEntityInterface
    {
        return $this
            ->createQuery()
            ->filterByAlias($alias)
            ->findOne();
    }
}