<?php

namespace Shop\Modules\LastOrdersWidget;

use ActiveTable\ColumnTable;
use ActiveTable\DataTableEngine;
use Core\Helpers\Date;
use DiamondTable\TableFactory;
use Shop\Config\Main;
use Shop\Domain\CustomerOrder\Contracts\CustomerOrderEntityInterface;
use Shop\Helpers\Formatter;
use Shop\Helpers\Price;
use Shop\Repositories\CustomerOrder\CustomerOrderCriteria;
use Shop\Repositories\CustomerOrder\CustomerOrderRepository;

/**
 * Description of LastOrdersWidget
 *
 * @author Diamond Code Generator
 */
class LastOrdersWidget extends \Core\ModuleController
{
    use \Shop\Modules\Traits\FormatterTrait;

    protected $orderRepository;
    protected $tableFactory;
    protected $mainConf;

    public function __construct(CustomerOrderRepository $orderRepository,
                                TableFactory $tableFactory,
                                Main $mainConf)
    {
        $this->orderRepository = $orderRepository;
        $this->tableFactory = $tableFactory;
        $this->mainConf = $mainConf;
    }

    public function process(): string
    {
        $cur = $this->mainConf->getCurencyDisplay();

        $criteria = CustomerOrderCriteria::create()
            ->setSortById('DESC')
            ->setFilterByStateId($this->mainConf->getStateDefault())
        ;

        $table = $this->tableFactory->buildAdminTable($this->orderRepository,$criteria);
        $table->removeControlAccess(DataTableEngine::CONTROL_ROWS_SELECT);
        $table->removeControlAccess(DataTableEngine::CONTROL_ROWS_ACTION);
        $table->removeControlAccess(DataTableEngine::CONTROL_ACCESS_ADD);
        $table->removeControlAccess(DataTableEngine::CONTROL_PAGINATION);

        $table
            ->addColumn(
                (new ColumnTable('orderCode','Номер заказа'))
                    ->setFormat($this,'columnCode')
                    ->setWidth(10,true)
            )
            ->addColumn(
                (new ColumnTable('datetime','Дата заказа'))
                    ->setFormat($this,'columnDateTime')
                    ->setWidth(10,true)
            )
            ->addColumn(
                (new ColumnTable('customer','Клиент/Группа клиента/Точка'))
                    ->setFormat($this,'formatCustomer')
                    ->setWidth(30,true)
            )
            ->addColumn(
                (new ColumnTable('discount','Скидка, ' . $cur))
                    ->setFormat($this,'formatDiscount')
                    ->setWidth(10,true)
            )
            ->addColumn(
                (new ColumnTable('orderFinalSumm','Сумма заказа, ' . $cur))
                    ->setFormat($this,'formatSumm')
                    ->setWidth(10,true)
            )
            ->addColumn(
                (new ColumnTable('itemsCount','Товаров'))
                    ->setWidth(10,true)
            )
            ->addColumn(
                (new ColumnTable('state','Состояние'))
                    ->setFormat($this,'formatStates')
                    ->setWidth(10,true)
            )
            ;

        return $this->render([
            'table' => $table->render()
        ]);
    }

    /**
     * @param CustomerOrderEntityInterface $order
     * @return string
     */
    public function formatSumm(CustomerOrderEntityInterface $order)
    {
        return Price::format($order->getOrderFinalSumm());
    }

    /**
     * @param CustomerOrderEntityInterface $order
     * @return string
     */
    public function formatDiscount(CustomerOrderEntityInterface $order)
    {
        return sprintf(
            '%s (%s)', Price::format($order->getDiscountSumm()), $order->getDiscountPercent() . '%'
        );
    }



    /**
     * @param \Shop\Models\Order\Order $order
     * @return \DateTime|string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnDateTime(\Shop\Models\Order\Order $order):string
    {
        return $order->getDatetime(Date::formatDisplay());
    }



    /**
     * @param \Shop\Models\Order\Order $order
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnCode(\Shop\Models\Order\Order $order):string
    {
        return anchor('/admin/shop/customers/orders?filter', $order->getOrderCode(), [], ['filter_by_ordercode' => $order->getOrderCode()]);
    }

}