<?php

namespace Shop\Modules\AdminPriceListsAccess;

use ActiveTable\ColumnTable;
use Core\Form\ControlBuilder;
use Diamond\Helpers\Form;
use DiamondTable\TableFactory;
use Shop\Domain\PriceAccess\Contracts\PriceAccessEntityInterface;
use Shop\Repositories\Customer\CustomerRepository;
use Shop\Repositories\CustomerGroup\CustomerGroupCriteria;
use Shop\Repositories\CustomerGroup\CustomerGroupRepository;
use Shop\Repositories\PriceAccess\PriceAccessCriteria;
use Shop\Repositories\PriceAccess\PriceAccessRepository;
use Shop\Repositories\PriceList\PriceListCriteria;
use Shop\Repositories\PriceList\PriceListRepository;
use Shop\Repositories\Stock\StockCriteria;
use Shop\Repositories\Stock\StockRepository;

/**
 * Description of AdminPriceListsAccess
 *
 * @author Diamond Code Generator
 */
class AdminPriceListsAccess extends \Core\ModuleController
{

    protected $priceAccessRepository;
    protected $tableFactory;
    protected $controlBuilder;
    protected $customerRepository;
    protected $stockRepository;
    protected $customerGroupRepository;
    protected $priceListRepository;

    public function __construct(PriceAccessRepository $priceAccessRepository, TableFactory $tableFactory,
                                ControlBuilder $controlBuilder, CustomerRepository $customerRepository,
                                StockRepository $stockRepository, CustomerGroupRepository $customerGroupRepository,
                                PriceListRepository $priceListRepository)
    {
        $this->priceAccessRepository = $priceAccessRepository;
        $this->tableFactory = $tableFactory;
        $this->controlBuilder = $controlBuilder;
        $this->customerRepository = $customerRepository;
        $this->stockRepository = $stockRepository;
        $this->customerGroupRepository = $customerGroupRepository;
        $this->priceListRepository = $priceListRepository;
    }

    public function process(): string
    {
        $table = $this->tableFactory->buildAdminTable($this->priceAccessRepository, PriceAccessCriteria::create());

        $table
            ->addColumn(
                (new ColumnTable('id', '№'))->setWidth(40)
            )
            ->addColumn(
                (new ColumnTable('customerGroupId', 'Группа клиента'))
                    ->setFormat($this, 'formatCustomerGroup')
            )
            ->addColumn(
                (new ColumnTable('customerId', 'Клиент'))
                    ->setFormat($this, 'formatCustomer')
            )
            ->addColumn(
                (new ColumnTable('shopId', 'Торговая точка'))
                    ->setFormat($this, 'formatStock')
            )
            ->addColumn(
                (new ColumnTable('priceScheetId', 'Прайс-лист'))
                    ->setFormat($this, 'formatPriceList')
            )
            ->addColumn(
                (new ColumnTable('actions', ''))
                    ->setFormat($this->tableFactory, 'formatActions')
                    ->setWidth(120)
            );

        $groups = $this->customerGroupRepository->findByCriteria(
            CustomerGroupCriteria::create()
        );

        $prices = $this->priceListRepository->findByCriteria(
            PriceListCriteria::create()
        );

        $stocks = $this->stockRepository->findByCriteria(
            StockCriteria::create()->setFilterByActive(true)->setSortByTitle('asc')
        );

        $caption = 'любой клиент';

        if (input_get('id')) {
            $access = $this->priceAccessRepository->findById(input_get('id'));

            if ($access->getCustomerId() > 0) {
                $customer = $this->customerRepository->findById($access->getCustomerId());
                $caption = $customer->getName();
            }

        }

        $table
            ->addField(
                $this->controlBuilder->buildAutoDropdown(
                    'priceScheetId',
                    Form::prepareDropdownWithDef($prices, '%','любой прайс-лист', 'getDropdownListWithFileName')
                )
            )
            ->addField(
                $this->controlBuilder->buildDropdownUniform('customerGroupId',
                    Form::prepareDropdownWithDef($groups,'%','любая группа клиента'))
            )
            ->addField(
                $this->controlBuilder->buildAutoDropdown(
                    'shopId',
                    Form::prepareDropdownWithDef($stocks, '%','любая торговая точка')
                )
            )
            ->addField(
                $this->controlBuilder->buildWindowSelect(
                    'customerId',
                    '/admin/shop/customers/customers-lite',
                    $caption
                )->setValue('%')
            );

        return $table->render();
    }

    /**
     * @param PriceAccessEntityInterface $accessEntity
     * @return string
     */
    public function formatCustomerGroup(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getCustomerGroupId() > 0) {
            $group = $this->customerGroupRepository->findById($accessEntity->getCustomerGroupId());
            return $group->getName();
        }
        return 'любая';
    }

    public function formatPriceList(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getPriceScheetId() > 0) {
            $price = $this->priceListRepository->findById($accessEntity->getPriceScheetId());
            return $price->getTitle();
        }
        return 'любой';
    }

    /**
     * @param PriceAccessEntityInterface $accessEntity
     * @return string
     */
    public function formatCustomer(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getCustomerId() > 0) {
            $customer = $this->customerRepository->findById($accessEntity->getCustomerId());
            return $customer->getName();
        }
        return 'любой';
    }

    public function formatStock(PriceAccessEntityInterface $accessEntity): string
    {
        if ($accessEntity->getShopId() > 0) {
            $shop = $this->stockRepository->findById($accessEntity->getShopId());
            return $shop->getTitle();
        }
        return 'любая';
    }

}               