<?php

namespace Shop\Models\Shop;

use Diamond\Domain\Site\Contracts\SiteEntityInterface;
use Diamond\Domain\Site\Contracts\SiteGetterInterface;
use Shop\Domain\Stock\Contracts\StockEntityInterface;
use Shop\Models\Shop\Base\Shop as BaseShop;

/**
 * Skeleton subclass for representing a row from the 'shop_shops' table.
 *
 *
 *
 * You should add additional methods to this class to meet the
 * application requirements.  This class will only be generated as
 * long as it does not already exist in the output directory.
 *
 */
class Shop extends BaseShop implements StockEntityInterface, SiteGetterInterface
{

    public const MSG_SHOP_MAIN = 'Головной офис';
    public const MSG_SHOP_DEALER = 'Филиал';
    public const MSG_SHOP_STOCK = 'Торговая точка';
    public const MSG_SHOP_PROVIDER = 'Поставщик';

    protected $stockChilds = 0;

    /**
     * @param int $v
     * @return $this|mixed
     */
    public function setCountChildrens(int $v)
    {
        $this->stockChilds = $v;
        return $this;
    }

    public function getCountChildrens(): int
    {
        return $this->stockChilds;
    }


    public function getSite(): SiteEntityInterface
    {
        return $this->getDomain();
    }


    /**
     * Полный адресс
     * @return type
     */
    public function getFullAddress(): string
    {

        $address = $this;

        if (!$city = $address->getCity()) {
            return "";
        }

        $region = $city->getCityRegion();
        return trim(trim(sprintf(
            "%s %s, %s, г.%s %s %s %s",
            $region->getRegionCountry()->getName(),
            $address->getPostCode(),
            $region->getName(),
            $city->getName(),
            $this->buildStreet(),
            $address->getHouse(),
            ($address->getRoom() ? ", кв. " . $address->getRoom() : "")
        ), ","));
    }

    public function getFullAddressCompact(): string
    {

        $address = $this;

        if (!$city = $address->getCity()) {
            return "";
        }

        $region = $city->getCityRegion();
        return trim(trim(sprintf(
            "%s г.%s %s %s %s",
            $region->getRegionCountry()->getName(),
            $city->getName(),
            $this->buildStreet(),
            $address->getHouse(),
            $this->buildOffice()
        ), ","));
    }

    /**
     * @return string
     */
    private function buildOffice(): string
    {
        $pref = mb_strpos($this->getRoom(), 'кв.') !== false || mb_strpos($this->getRoom(), 'офис') !== false ? '' : 'кв.';
        return $this->getRoom() ? ", " . $pref . $this->getRoom() : "";
    }

    /**
     * @return string
     */
    private function buildStreet(): string
    {
        $pref = mb_strpos($this->getStreet(), 'ул.') !== false || mb_strpos($this->getStreet(), 'улица') !== false ? '' : 'ул.';
        return $this->getStreet() > "" ? ", " . $pref . $this->getStreet() : "";
    }

    public function getFullAddressCompactWithoutCountry(): string
    {

        $address = $this;

        if (!$city = $address->getCity()) {
            return "";
        }

        $region = $city->getCityRegion();
        return trim(trim(sprintf(
            "г.%s%s %s",
            $city->getName(),
            $address->getStreet() > "" ? ", ул." . $address->getStreet() : "", $address->getHouse(), $address->getRoom() ? ", кв. " . $address->getRoom() : ""
        ), ","));
    }

    /**
     * Тип точки для отображения
     * @return string
     */
    public function getTypeDisplay(): string
    {
        switch ($this->getType()) {
            case 'main':

                return self::MSG_SHOP_MAIN;

                break;

            case 'dealer':

                return self::MSG_SHOP_DEALER;

                break;

            default:

                return self::MSG_SHOP_STOCK;

                break;
        }
    }

    /**
     * Title + type
     * @return string
     */
    public function getTitleWithType(): string
    {
        return sprintf(
            "%s (%s)",
            $this->getTitle(),
            $this->getTypeDisplay()
        );
    }

}