<?php

namespace Shop\Modules\AdminStateLogs;

use ActiveTable\ColumnTable;
use ActiveTable\DataTableEngine;
use Core\Form\ControlBuilder;
use DiamondTable\TableFactory;
use Shop\Domain\StateLog\Contracts\StateLogEntityInterface;
use Shop\Helpers\Formatter;
use Shop\Models\StateLog\StateLog;
use Shop\Repositories\CustomerOrder\CustomerOrderRepository;
use Shop\Repositories\State\StateCriteria;
use Shop\Repositories\State\StateRepository;
use Shop\Repositories\StateLog\StateLogCriteria;
use Shop\Repositories\StateLog\StateLogRepository;

/**
 * Description of AdminStateLogs
 *
 * @author Diamond Code Generator
 */
class AdminStateLogs extends \Core\ModuleController
{

    protected $stateLogRepository;
    protected $tableFactory;
    protected $controlBuilder;
    protected $customerOrderRepository;
    protected $stateRepository;
    protected $states = [];

    /**
     * AdminStateLogs constructor.
     * @param StateLogRepository $stateLogRepository
     * @param TableFactory $tableFactory
     * @param ControlBuilder $controlBuilder
     * @param CustomerOrderRepository $customerOrderRepository
     * @param StateRepository $stateRepository
     */
    public function __construct(StateLogRepository $stateLogRepository, TableFactory $tableFactory,
                                ControlBuilder $controlBuilder, CustomerOrderRepository $customerOrderRepository,
                                StateRepository $stateRepository)
    {
        $this->stateLogRepository = $stateLogRepository;
        $this->tableFactory = $tableFactory;
        $this->controlBuilder = $controlBuilder;
        $this->customerOrderRepository = $customerOrderRepository;
        $this->stateRepository = $stateRepository;
    }

    public function process(): string
    {
        $states = $this->stateRepository->findByCriteria(
            StateCriteria::create()
        );
        foreach ($states as $state) {
            $this->states[$state->getId()] = $state;
        }

        $table = $this->tableFactory->buildAdminTable($this->stateLogRepository, StateLogCriteria::create());

        $table
            ->removeControlAccess(DataTableEngine::CONTROL_ROWS_ACTION)
            ->removeControlAccess(DataTableEngine::CONTROL_ACCESS_ADD)
            ->setDefaultSortColumn('id', 'desc')
            ->addColumn(
                (new ColumnTable('id', '№'))->setWidth(40)
            )
            ->addColumn(
                (new ColumnTable('datetime', 'Дата'))
                    ->setFormat($this, 'formatDate')

            )
            ->addColumn(
                (new ColumnTable('oldStateId', 'Старое состояние'))
                    ->setFormat($this, 'formatStateOld')

            )
            ->addColumn(
                (new ColumnTable('newStateId', 'Новое состояние'))
                    ->setFormat($this, 'formatStateNew')

            )
            ->addColumn(
                (new ColumnTable('objectId', '№ заказа'))
                    ->setFormat($this, 'formatCode')
            )
            ->addColumn(
                (new ColumnTable('actions', ''))
            )
        ;

        $table->addFilter(
            $this->controlBuilder->buildInput('filter_by_objectId')
        );

        return $table->render();
    }


    /**
     * @param StateLogEntityInterface $log
     * @return string
     */
    public function formatDate(StateLogEntityInterface $log): string
    {
        return $log->getDatetime()->format(\Diamond\Helpers\Date::formatDisplay());
    }

    /**
     * @param StateLogEntityInterface $log
     * @return string
     */
    public function formatCode(StateLogEntityInterface $log): string
    {
        if ((int)$log->getObjectId() === 0) {
            return '';
        }

        if ($log->getTypeId() === StateLog::TYPE_ORDER) {
            return 'Заказа №' . $log->getObjectId();
        } elseif ($log->getTypeId() === StateLog::TYPE_ORDER_ITEM) {
            return 'Позиция заказа №' . $log->getObjectId();
        }

        return '';
    }

    /**
     * @param StateLogEntityInterface $log
     * @return string
     */
    public function formatStateOld(StateLogEntityInterface $log): string
    {
        if ((int)$log->getOldStateId() === 0) {
            return '';
        }


        return Formatter::formatColor($this->states[$log->getOldStateId()]);

    }

    /**
     * @param StateLogEntityInterface $log
     * @return string
     */
    public function formatStateNew(StateLogEntityInterface $log): string
    {
        if ((int)$log->getNewStateId() === 0) {
            return '';
        }


        return Formatter::formatColor($this->states[$log->getNewStateId()]);

    }
}