<?php
namespace Shop\Repositories;

use Core\Repository\CrudInterface;
use Core\Repository\PropelQueryAbstract;
use Propel\Runtime\ActiveQuery\Criteria;
use Propel\Runtime\Collection\ObjectCollection;
use Shop\Models\Category\Category as CategoryEntity;
use Shop\Models\Category\CategoryQuery;
use Shop\Contracts\CatalogFilterInterface;

/**
 * Description of Category
 *
 * @author Dmitriy
 */
class Category extends PropelQueryAbstract implements CrudInterface, \Core\Repository\DropdownInreface{

    public function buildEntity(): CategoryEntity{
	return new CategoryEntity;
    }

    public function buildQuery(): CategoryQuery{
	return CategoryQuery::create();
    }
    
    /**
     * получение списка
     * @return type
     */
    public function getDropdownList(){
	return $this->buildQuery()
		->orderByTitle()
		->find()
		->toKeyValue('id', 'title');
    }
    
    /**
     * получение списка
     * @return type
     */
    public function getDropdownListWithParent(){
	return $this->buildQuery()
		->orderByParentId()
                ->orderByTitle()
		->find()
		->toKeyValue('id', 'NameWithParentName');
    }  
    
    /**
     * получение списка
     * @return type
     */
    public function getDropdownListAllWithParent(){
	return $this->buildQuery()
		->orderByParentId()
                ->orderByTitle()
                //->filterByParentId(0, Criteria::GREATER_THAN)
		->find()
		->toKeyValue('id', 'nameWithParent');
    }     
    
    /**
     * 
     * @param type $parent
     * @return ObjectCollection
     */
    public function findByParentId($parent = null) : ObjectCollection {
        return $this->buildQuery()
		->orderByTitle()
                ->filterByParentId($parent)
		->find();
    }
    
    /**
     * по ключам
     * @param array $ids
     * @return type
     */
    public function findByIds(array $ids)
    {
         return $this->buildQuery()
		->orderByTitle()
                ->filterByPrimaryKeys($ids)
		->find();
    }
    
    /**
     * Получение разделов только с товаров который имеет фото
     *
     * @param \Shop\Contracts\CustomerAccessPricesInterface $customer
     * @param type $parent
     * @return ObjectCollection
     */
    public function findByParentIdWithImage(\Shop\Contracts\CustomerAccessPricesInterface $customer, $parent = null) : ObjectCollection {
        
        
        return $this->buildQuery()
		->orderByTitle()
                ->filterByParentId($parent)
                ->useProductQuery()
                    ->filterByImg('', Criteria::GREATER_THAN)
                    ->usePositionQuery()
                           ->filterByPricelistSheetId($customer->getAccessPriceListScheetIds())
                           ->filterByRemain(0,Criteria::GREATER_THAN)
                    ->endUse()
                ->endUse()
                ->groupById()
		->find();
    }
    
    /**
     * @todo перевсти на корректный запрос по скидкам
     * Получение разделов с скидочными товарами
     * @param type $parent
     * @return ObjectCollection
     */
    public function findByParentIdWithImageAndSale(\Shop\Contracts\CustomerAccessPricesInterface $customer, $parent = null) : ObjectCollection {
        return $this->buildQuery()
		->orderByTitle()
                ->filterByParentId($parent)
                ->useProductQuery()
                    ->filterByImg('', Criteria::GREATER_THAN)
                    ->usePositionQuery()
                           ->filterByPricelistSheetId($customer->getAccessPriceListScheetIds())
                           ->filterByRemain(0,Criteria::GREATER_THAN)
                           ->filterBySaleId(1)
                    ->endUse()                
                ->endUse()
                ->groupById()
		->find();        
    }
    
   /**
    * получение разделов верхнего уровня
    * @param \Shop\Contracts\CustomerAccessPricesInterface $customer
    * @param type $days
    * @param type $parent
    * @return ObjectCollection
    */
    public function findByParentIdWithImageAndNew(\Shop\Contracts\CustomerAccessPricesInterface $customer,$days, $parent = null) : ObjectCollection {
        return $this->buildQuery()
		
                ->filterByParentId($parent)
                ->useCategoryRelatedByIdQuery("parent")
               
                    ->useProductQuery()
                        ->filterBy("dateCreate", sprintf( "cdate >= DATE_SUB(CURRENT_DATE, INTERVAL %s DAY)",$days ), Criteria::CUSTOM)
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->usePositionQuery()
                               ->filterByPricelistSheetId($customer->getAccessPriceListScheetIds())
                               ->filterByRemain(0,Criteria::GREATER_THAN)
                        ->endUse()                
                    ->endUse()
                ->endUse()
                ->groupById()
                ->orderByTitle()
		->find();        
    }    
    
    /**
     * Поиск раздела по всевдониму
     * @param type $param
     */
    public function findOneByAlias($alias, $ignoreRootCategory = true) {
        
        return $this->buildQuery()
		->filterByAlias($alias)
                ->_if($ignoreRootCategory===true)
                    ->filterByParentId(0, Criteria::GREATER_THAN)
                ->_endif()
		->findOne();
    }
    
    /**
     * 
     * @param type $alias
     * @return type
     */
    public function findByParentAlias($alias) {

        return $this->buildQuery()
                ->useCategoryRelatedByIdQuery("parent")
                 ->filterByParentId(0,Criteria::GREATER_THAN)
                ->endUse()
                ->filterByAlias($alias)
		->find();
    }    
}
