<?php

namespace Shop\Repositories;

use Core\Repository\CrudInterface;
use Core\Repository\DropdownInreface;
use Core\Repository\PropelQueryAbstract;
use Propel\Runtime\ActiveQuery\Criteria;
use Shop\Models\Brand\Brand as BrandEntity;
use Shop\Models\Brand\BrandQuery;
use Shop\Contracts\CatalogFilterInterface;

/**
 * Description of Brand
 *
 * @author Dmitriy
 */
class Brand extends PropelQueryAbstract implements CrudInterface, DropdownInreface
{

    public function buildEntity()
    {
        return new BrandEntity;
    }

    public function buildQuery()
    {
        return BrandQuery::create();
    }

    public function getDropdownList()
    {
        return BrandQuery::create()
                        ->orderByName()
                        ->find()
                        ->toKeyValue('id', 'name');
    }

    /**
     * Получаем производителей по категории продуктов
     * @param type $category_id
     */
    public function findBrandsByProductCategory($category_alias, $brand_alias = null)
    {
        return $this
                        ->buildQuery()
                        ->useProductQuery()
                        ->useCategoryQuery()
                        ->filterByAlias($category_alias)
                        ->endUse()
                        ->endUse()
                        ->_if($brand_alias)
                        ->filterByAlias($brand_alias)
                        ->_endif()
                        ->groupById()
                        ->find();
    }

    /**
     * Получение брендов с изображениями с игнором по брендом и с привязкой к магазу
     * @param type $shopId
     * @param type $ignore
     */
    public function findWithImagesByShopIdWithoutHideBrands($shopId, $ignore, array $orderAlias = [], $limit = false)
    {
        
        return $this
                        ->buildQuery()
                        ->useProductQuery()
                        ->filterByImg('', Criteria::GREATER_THAN)
                            ->usePositionQuery()
                            ->filterByRemain(0, Criteria::GREATER_THAN)
                                ->usePricelistSheetQuery()
                                    ->usePricelistFileQuery()
                                        ->useProviderQuery()
                                            ->filterByShopId($shopId)
                                        ->endUse()
                                    ->endUse()
                                 ->endUse()
                            ->endUse()
                        ->endUse()
                        ->filterById($ignore, Criteria::NOT_IN)
                        ->filterByAlias('', Criteria::GREATER_THAN)
                        ->groupById()
                        ->_if(count($orderAlias))
                        ->addDescendingOrderByColumn(sprintf(
                                        "FIELD(`shop_brands`.`alias` ,'%s')", implode("','", $orderAlias)
                        ))
                        ->_endif()
                        ->orderByAlias()
                        ->addAscendingOrderByColumn('rand()')
                        ->_if($limit)
                        ->limit($limit)
                        ->_endif()
                        ->find();
    }

    /**
     * 
     * @param type $title
     * @return type
     */
    public function findByName($title, CatalogFilterInterface $catalogFilter)
    {
        return $this
                        ->buildQuery()
                        ->filterByName("{$title}%", Criteria::LIKE)
                        ->orderByName()
                        ->useProductQuery()
                            ->usePositionQuery()
                                ->filterByRemain(0, Criteria::GREATER_THAN)
                                ->filterByPricelistSheetId($catalogFilter->getPriceListScheetIds())
                            ->endUse()
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->endUse()
                        ->limit(10)
                        ->groupById()
                        ->find()
        ;
    }

    /**
     * 
     * @param type $alias
     * @return type
     */
    public function findOneByAlias($alias)
    {
        return $this
                        ->buildQuery()
                        ->filterByAlias($alias)
                        ->findOne();
    }
    
    /**
     * Получение брендов с изображениями с игнором по брендом и с привязкой к магазу
     * @param type $shopId
     * @param type $ignore
     */
    public function findWithImagesByCustomerAccess(CatalogFilterInterface $catalogFilter)
    {
        
        return $this
                ->buildQuery()
                ->useProductQuery()
                ->filterByImg('', Criteria::GREATER_THAN)
                    ->usePositionQuery()
                    ->filterByRemain(0, Criteria::GREATER_THAN)
                    ->filterByImage('', Criteria::GREATER_THAN)
                    ->filterByPricelistSheetId($catalogFilter->getPriceListScheetIds())
                    ->_if($catalogFilter->getSale() === true)
                        ->filterBySaleId(1)
                    ->_endif() 
                    ->endUse()
                ->endUse()
                ->groupById()
                ->orderByName()
                ->find();
    }    

}