<?php

namespace Shop\Repositories;

use Core\Repository\CrudInterface;
use Core\Repository\PropelQueryAbstract;
use DateTime;
use Propel\Runtime\ActiveQuery\Criteria;
use Propel\Runtime\Collection\ObjectCollection;
use Shop\Models\Product\Product as ProductEntity;
use Shop\Models\Product\ProductQuery;
use Shop\Contracts\CatalogFilterInterface;
use Propel\Runtime\Util\PropelModelPager;

/**
 * Description of Bonus
 *
 * @author Dmitriy
 */
class Product extends PropelQueryAbstract implements CrudInterface
{

    public function buildQuery()
    {
        return ProductQuery::create()
                //->leftJoinProductSuperPrice("superPrice")
                //->leftJoinProductHitSale("hitSale")
                //->addAsColumn('superPrice', 'superPrice.Image')
                //->addAsColumn('hitSale', 'hitSale.Image')
                ;
    }

    public function buildEntity()
    {
        return new ProductEntity;
    }

    /**
     * Получить по бренду
     * @param type $brand_id
     * @return type
     */
    public function findProductByBrandId($brand_id, $page = 1, $maxPerPage = 10)
    {

        return $this->buildQuery()
                        ->filterByBrandId($brand_id)
                        ->paginate($page, $maxPerPage);
    }

    /**
     * @todo добавить фильтр по наличию
     * @param type $id
     * @return type
     */
    public function countByBrandId($id)
    {
        return $this->buildQuery()
                        ->filterByBrandId($id)
                        ->count();
    }

    /**
     * 
     * @param type $id
     * @return type
     */
    public function countByCategoryId(CatalogFilterInterface $catalogFilter)
    {
        return $this->buildQuery()
                        ->filterByCategoryId($catalogFilter->getCategoryId())
                        ->usePositionQuery()
                        ->filterByRemain(0, Criteria::GREATER_THAN)
                        ->filterByShopId($catalogFilter->getShopId())
                   
                        
                            ->_if($catalogFilter->getSale() === true)
                                ->filterBySaleId(1)
                            ->_endif() 
                                        
                        ->endUse()
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->groupById()
                        ->count();
    }

    /**
     * 
     * @param type $shopId
     * @param type $categ_id
     * @param type $page
     * @param type $maxPerPage
     * @return type
     */
    public function findProductByCategoryId(CatalogFilterInterface $catalogFilter, $page = 1, $maxPerPage = 10,
                                            $sort = null)
    {
        //dump($catalogFilter);
        $sizeTypes = (array) $catalogFilter->getSizeType();
        $sizeValues = [];
        foreach ($sizeTypes as $type) {
            $sizeValues = array_merge($catalogFilter->getSizeValues()[$type], $sizeValues);
        }
        
        //dump($catalogFilter);exit;
        $q = $this->buildMainCatalogQuery($catalogFilter)

                //positions
                ->usePositionQuery()
                ->_if($sort == "first-cheap")
                ->orderByPrice(Criteria::ASC)
                ->_endif()
                ->_if($sort == "first-roads")
                ->orderByPrice(Criteria::DESC)
                ->_endif()
                ->_if($catalogFilter && $catalogFilter->getSizeValues() && $catalogFilter->getSizeType())
                ->useShopPositionSizeQuery()
                ->filterBySizeTypeId($sizeTypes)
                ->filterByValue($sizeValues)
                ->endUse()
                ->_endif()
                ->_if($catalogFilter->getPriceMin() > 0)
                ->filterByPrice($catalogFilter->getPriceMin(), Criteria::GREATER_EQUAL)
                ->_endif()
                ->_if($catalogFilter->getPriceMax() > 0)
                ->filterByPrice($catalogFilter->getPriceMax(), Criteria::LESS_EQUAL)
                ->_endif()              
                ->endUse()
                //positions - end                
                ->_if($catalogFilter && count($catalogFilter->getBrands()) > 0)
                ->filterByBrandId($catalogFilter->getBrands())
                ->_endif()
                ->_if($catalogFilter && count($catalogFilter->getGroups()) > 0)
                ->useShopProductGroupsRelQuery()
                ->filterByGroupId($catalogFilter->getGroups())
                ->endUse()
                ->_endif()
                ->_if($sort == "news")
                ->orderBydateCreate(Criteria::DESC)
                ->_endif()
                ->_if($sort == "name-a-z")
                ->orderByTitle(Criteria::ASC)
                ->_endif()
                ->_if($sort == "name-z-a")
                ->orderByTitle(Criteria::DESC)
                ->_endif()
                ->groupById()

        ;
         // echo $q->toString();
           //exit;
        return $q->paginate($page, $maxPerPage);
    }

    /**
     * 
     * @param type $shopId
     * @param type $categ_id
     */
    public function findBrandsByCategoryId(CatalogFilterInterface $catalogFilter)
    {
        $query1 = $this->buildMainCatalogQuery($catalogFilter)
                ->useBrandQuery()
                ->withColumn("shop_brands.name", "brand_name")
                ->endUse()
                ->select(["brand_name"])
                ->addSelfSelectColumns(true)
                ->groupById()
        ;

        $query2 = ProductQuery::create()
                ->addAsColumn('brand_name', 'prod.brand_name')
                ->addSelectQuery($query1, 'prod')
                ->addAsColumn('count', 'COUNT(prod.id)')
                ->groupByBrandId()
                ->orderBy("prod.brand_name")
        ;

        return $query2->find();
    }

    /**
     * 
     * @param CatalogFilterInterface $catalogFilter
     * @return type
     */
    private function buildMainCatalogQuery(CatalogFilterInterface $catalogFilter)
    {

        return $this->buildQuery()
                        ->_if($catalogFilter->getNew() > 0)
                        //->filterBydateCreate("DATE_SUB(CURRENT_DATE, INTERVAL 30 DAY)",Criteria::GREATER_EQUAL)
                        ->filterBy("dateCreate", sprintf( "cdate >= DATE_SUB(CURRENT_DATE, INTERVAL %s DAY)",$catalogFilter->getNew() ), Criteria::CUSTOM)
                        ->_endif()                 
                        ->usePositionQuery()
                        ->filterByRemain(0, Criteria::GREATER_THAN)
                        ->filterByShopId($catalogFilter->getShopId())
                            ->_if($catalogFilter->getSale() === true)
                                ->filterBySaleId(1)
                            ->_endif() 
                        ->endUse()
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->_if($catalogFilter->getCategoryId())
                        ->filterByCategoryId($catalogFilter->getCategoryId())
                        ->_endIf()
                        ->_if($catalogFilter->getProductName())
                        ->filterByTitle($catalogFilter->getProductName())
                        ->_endIf()
                        ->_if($catalogFilter->getBrandName())
                        ->useBrandQuery()
                        ->filterByName($catalogFilter->getBrandName())
                        ->endUse()
                        ->_endIf()
                        ->_if(count($catalogFilter->getProducts()))
                        ->filterByPrimaryKeys($catalogFilter->getProducts())
                        ->_endIf()
                        
        ;
    }

    /**
     * Получение товаров с группировкой по сезонам
     * @param CatalogFilterInterface $catalogFilter
     * @return type
     */
    public function findSeazonsByCategoryId(CatalogFilterInterface $catalogFilter,array $groups)
    {
        return $this->buildMainCatalogQuery($catalogFilter)
                        ->useShopProductGroupsRelQuery()
                        ->_if(count($groups))
                        ->filterByGroupId($groups)
                        ->_endif()
                        ->groupByGroupId()
                        ->endUse()
                        ->find()
        ;
    }

    /**
     * 
     * @param CatalogFilterInterface $catalogFilter
     * @return type
     */
    public function findSizesByCategoryIdAndTypeId(CatalogFilterInterface $catalogFilter)
    {
        return $this->buildMainCatalogQuery($catalogFilter)
                        ->usePositionQuery()
                        ->useShopPositionSizeQuery()
                        ->groupByValue()
                        ->useShopPositionSizeTypeQuery()
                        ->filterById($catalogFilter->getSizeType())
                        ->endUse()
                        ->endUse()
                        ->endUse()
                        ->withColumn("shop_position_sizes.value", "color_name")
                        ->select('color_name')
                        ->find()
        ;
    }

    /**
     * 
     * @param CatalogFilterInterface $catalogFilter
     * @return type
     */
    public function findMinMaxPricesByCategoryId(CatalogFilterInterface $catalogFilter)
    {

        $c = new Criteria();
        $c->addSelectColumn('MAX(price) as min', Criteria::CUSTOM);

        return $this->buildMainCatalogQuery($catalogFilter)
                        ->select('max_price')
                        ->addAsColumn('max_price', 'MAX(shop_positions.price)')
                        ->select('min_price')
                        ->addAsColumn('min_price', 'MIN(shop_positions.price)')
                        ->findOne()
        ;
    }

    /**
     * 
     * @return ObjectCollection
     */
    public function findAll(): ObjectCollection
    {
        return $this->buildQuery()
                        ->find();
    }

    /**
     * 
     * @param type $page
     * @param type $maxPerPage
     * @return PropelModelPager
     */
    public function findAllByPaginate($page = 1, $maxPerPage = 10): PropelModelPager
    {
        return $this->buildQuery()
                        ->paginate($page, $maxPerPage);
    }

    /**
     * Получение акционных по лимиту
     * @return type
     */
    public function findProductsSales($limit = 10, $categoryId = null): ObjectCollection
    {

        return $this->buildQuery()
                        ->filterBySpecial("Y")
                        ->limit($limit)
                        ->addAscendingOrderByColumn('rand()')
                        ->find();
    }

    /**
     * Получение новинок продуктов для раздела
     * 
     * @param type $categoryId
     * @param type $limit
     * @param type $minDate
     * @return ObjectCollection
     */
    public function findNewsProductsByParentId(CatalogFilterInterface $catalogFilter,$parentId, $limit = 10, $daysLimit = 30): ObjectCollection
    {
        return $this->buildMainCatalogQuery($catalogFilter)
                        ->useCategoryQuery()
                        ->filterByParentId($parentId)
                        ->endUse()
                        ->limit($limit)
                        ->addAscendingOrderByColumn('rand()')
                        ->filterBydateCreate([
                            "min" => (new DateTime(sprintf("today -%s days", $daysLimit)))
                        ])
                        ->groupById()
                        ->find();
    }

    /**
     * Получение скидочных товаров по разделам
     * @param ObjectCollection $category
     * @param type $limit
     * @return ObjectCollection
     */
    public function findSalesProductsByCategories(CatalogFilterInterface $catalogFilter, $limit = 10): ObjectCollection
    {
        return $this->buildQuery()
                        ->useCategoryQuery()
                        ->filterByParentCategory($catalogFilter->getCategoryId())
                        ->endUse()
                        ->usePositionQuery()
                        ->filterByShopId($catalogFilter->getShopId())
                        ->filterByRemain(0, Criteria::GREATER_THAN)
                        ->endUse()
                        ->limit($limit)
                        ->addAscendingOrderByColumn('rand()')
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->groupById()
                        ->find();
    }

    /**
     * 
     * @param CatalogFilterInterface $catalogFilter
     * @param type $limit
     * @return type
     */
    public function findSimilarByCategoryId(CatalogFilterInterface $catalogFilter, $limit = 10)
    {
        return $this->buildMainCatalogQuery($catalogFilter)
                        ->limit($limit)
                        ->addAscendingOrderByColumn('rand()')
                        ->groupById()
                        ->find();
    }

    /**
     * 
     * @param type $param
     */
    public function findByName($title,$shopId)
    {
        return $this->buildQuery()
                        ->filterByTitle("{$title}%", Criteria::LIKE)
                        ->usePositionQuery()
                        ->filterByRemain(0, Criteria::GREATER_THAN)
                        ->filterByShopId($shopId)
                        ->endUse()
                        ->filterByImg('', Criteria::GREATER_THAN)
                        ->orderByTitle()
                        ->groupByTitle()
                        ->limit(10)
                        ->find();
    }

    /**
     * 
     * @param type $title
     * @return type
     */
    public function findByArticle($title,$shopId)
    {
        return $this->buildQuery()
                        ->usePositionQuery()
                        ->filterByRemain(0, Criteria::GREATER_THAN)
                        ->filterByShopId($shopId)
                        ->endUse()
                        ->filterByImg('', Criteria::GREATER_THAN)                
                        ->filterByArticle("{$title}%", Criteria::LIKE)
                        ->orderByArticle()
                        ->groupByArticle()
                        ->limit(10)
                        ->find();
    }
    
    /**
     * 
     * @param type $id
     * @return \Shop\Models\Product\Product
     */
    public function findById($id){
	return parent::findById($id);
    }

}