<?php


namespace MoySklad\Application\Commands\CreateOrder;


use MoySklad\Components\FilterQuery;
use MoySklad\Entities\Counterparty;
use MoySklad\Entities\Documents\Orders\CustomerOrder;
use MoySklad\Entities\Organization;
use MoySklad\Entities\Products\Product;
use MoySklad\Entities\Store;
use MoySklad\Exceptions\ApiResponseException;
use MoySklad\Lists\EntityList;
use MoySklad\MoySklad;
use Shop\Config\Main;
use Shop\Domain\CustomerOrder\Contracts\CustomerOrderEntityInterface;
use Shop\Models\Customer\Customer;
use Shop\Models\MoyskladSyncLogs\MoyskladSyncLogs;
use Shop\Models\Order\Order;
use Shop\Models\OrderItem\OrderItem;

/**
 * Создаем заказ на стороне сервиса МойСклада
 * Class CreateOrderHandler
 * @package MoySklad\Application\Commands\CreateOrder
 */
class CreateOrderHandler
{
    /**
     * @var Main
     */
    protected $mainConfig;

    /**
     * CreateOrderHandler constructor.
     * @param Main $mainConfig
     */
    public function __construct(Main $mainConfig)
    {
        $this->mainConfig = $mainConfig;
    }

    /**
     * @param CreateOrderCommand $command
     * @throws \MoySklad\Exceptions\EntityCantBeMutatedException
     * @throws \MoySklad\Exceptions\IncompleteCreationFieldsException
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function handle(CreateOrderCommand $command): void
    {
        /**
         * @var Order $order
         */
        $order = $command->getOrder();
        $storeName = $command->getRemoteStock();

        $organization = null;

        $contrEmail = $order->getContractor()->getEmail();

        $login = $this->mainConfig->getMoyskladLogin();
        $pass = $this->mainConfig->getMoyskladPassword();

        try {
            $conn = MoySklad::getInstance($login, $pass);
        } catch (ApiResponseException $ex) {
            $text = implode(',', $ex->getResponse()->errors);
            $this->addSyncLog($order, $text);
            throw new CreateOrderException($text);
        }

        // FIND POSITIONS----------------------------------------------
        $stocks = $warnings = $positions = [];

        $part = 0; $i=0;

        /**
         * @var OrderItem $item
         */
        foreach ($order->getOrderItems() as $item) {

            $stocks []= $item->getDestination();

            try {
                $product = null;
                $products = Product::query($conn)->filter(
                    (new FilterQuery())
                        ->eq("externalCode", $item->getRefId())
                );

                foreach ($products as $product) {
                    break;
                }

                if (!$product) {
                    $text = "product not found by externalCode " . $item->getRefId();

                    $warnings[] = $text;

                    $this->addSyncLog($order, $text);

                    continue;
                }

            } catch (ApiResponseException $ex) {
                $text = implode(',', $ex->getResponse()->errors);

                $warnings[] = $text;

                $this->addSyncLog($order, $text);

                continue;
            }


            $product->quantity = $item->getAmount();
            $product->price = (float)$item->getPrice() * 100;

            $i++;

            if($i > 100){
                $part++;
                $i=0;
            }

            $positions[$part][] = $product;
        }


        if(count($positions)===0){
            $text = 'not found positions for sync';
            $this->addSyncLog($order, $text);
            throw new CreateOrderException($text);
        }

        //ORDER COMMENT
        $stocks = array_unique($stocks);

        //к комменту добавляем название склада
        $description = implode(',', $stocks) . '. '. $order->getComment();
        // FIND POSITIONS END----------------------------------------------

        //если в позициях заказ один склад то найдем и привяжем заказ к этому складу
        if(count($stocks)===1 && !empty($stocks[0])){
            $storeName = $stocks[0];
        }

        // FIND STORE
        if($storeName){
            $stores = Store::query($conn)
                ->filter((new FilterQuery())
                    ->eq('name', $storeName
                    )
                );

            foreach ($stores as $store) {
                break;
            }

        }


        // FIND ORGANIZATION----------------------------------------------
        $contragents = Organization::query($conn)
            ->filter((new FilterQuery())
                ->eq('email', $contrEmail
                )
            );

        $this->addSyncLog($order, 'start order sync');

        /**
         * поиск организации
         * @var \MoySklad\Entities\Organization $item
         */
        foreach ($contragents as $organization) {
            break;
        }

        if (!$organization) {

            $text = 'organization not found by email ' . $contrEmail;

            $this->addSyncLog($order, $text);

            throw new CreateOrderException($text);
        }
        // FIND ORGANIZATION----------------------------------------------


        $email = $order->getCustomer()->getEmail();
        $customer = null;

        //поиск клиента по email
        $clients = Counterparty::query($conn)
            ->filter((new FilterQuery())
                ->eq('email', $email));

        foreach ($clients as $item) {
            $customer = $item;
            break;
        }


        if (!$customer) {
            /**
             * @var Customer $orderCustomer
             */
            $orderCustomer = $order->getCustomer();

            $customer = (new Counterparty($conn, [
                'name' => $orderCustomer->getName(),
                'email' => $email,
                'address' => $orderCustomer->getFullAddress(),
                'phone' => $orderCustomer->getPhone()
            ]))
                ->buildCreation()
                ->execute();


            $this->addSyncLog($order, 'create new customer, email ' . $email);

        }


        try {

            $ids = '';
            foreach ($positions as $items){

                $remoteOrder = (new CustomerOrder($conn,['description'=>$description]))->buildCreation();

                if(isset($store)){
                    $remoteOrder->addStore($store);
                }

                $remoteOrder = $remoteOrder
                    ->addOrganization($organization)
                    ->addCounterparty($customer)
                    ->addPositionList(new EntityList($conn, $items))
                    ->execute();

                $text = 'create remote order id:' . $remoteOrder->findEntityId();

                $this->addSyncLog($order, $text);
                $ids .= $remoteOrder->findEntityId().',';
            }

            $command->setResultId(rtrim($ids,','));

            if (count($warnings)) {
                throw new CreateOrderWarningException(implode(',', $warnings));
            }

        } catch (ApiResponseException $ex) {
            $error = implode(',', $ex->getResponse()->errors);
            $this->addSyncLog($order, $error);
            throw new CreateOrderException($error);
        }

    }

    /**
     * @param Order $order
     * @param string $text
     * @throws \Propel\Runtime\Exception\PropelException
     */
    private function addSyncLog(Order $order, string $text): void
    {
        $log = new MoyskladSyncLogs();
        $log->setDesc($text);
        $log->setOrder($order);
        $log->save();
    }

}