<?php
/**
 * Created by PhpStorm.
 * User: Admin
 * Date: 27.05.2020
 * Time: 0:26
 */

namespace MoySklad\Application;


use GuzzleHttp\Exception\ConnectException;
use Local\Config\Main;
use MoySklad\Application\Exceptions\MoySkladException;
use MoySklad\Application\VelueObjects\TaskStatus;
use MoySklad\Domain\MoySkladTask\Contracts\MoySkladEntityInterface;
use MoySklad\Models\MoySkladTask\MoySkladTask;
use GuzzleHttp\Client;

class MoySkladTaskService
{

    public const TASKS_FILES_DIR = '/app/cache/sync/';
    public const TASKS_PROCESS_FILE = 'data.process';
    public const TASKS_FINISH_FILE = 'data.finish';

    public const STATUS_WORK_CODE = 'work';
    public const STATUS_WAIT_CODE = 'waiting';
    public const STATUS_HOVERED_CODE = 'hovered';
    public const STATUS_FINISH_CODE = 'finish';

    protected $mainConfig;

    public function __construct(Main $mainConfig)
    {
        $this->mainConfig = $mainConfig;
    }

    /**
     * @param MoySkladEntityInterface $task
     */
    public function loadMoySkladTask(MoySkladEntityInterface $task): void
    {
        $client = new Client([
            'base_uri' => getenv('BASE_URL'),
            'timeout' => 5,
            //'connect_timeout' => 3,
            'allow_redirects' => false
        ]);


        //первый запрос на себя для загрузки пачками
        $res = $client->post('/_service/mysklad/remains/part/' . $task->getComplete(), [
            'form_params' => [
                'task' => $task->getId(),
                'mark' => $task->getMark()
            ]
        ]);


    }

    /**
     * @param MoySkladEntityInterface $task
     * @param $mark
     * @param MoySkladService $moySkladService
     * @throws \Propel\Runtime\Exception\PropelException
     * @throws \Throwable
     */
    public function loadMoySkladTaskFirst(MoySkladEntityInterface $task, $mark, MoySkladService $moySkladService): void
    {

        /**
         * @var MoySkladTask $task
         */
        $task->setMark($mark);
        $task->setDateTimeStart(new \DateTime());
        $task->setDateTimeEnd(null);
        $task->setComplete(0);
        $task->save();

        $client = new Client([
            'base_uri' => getenv('BASE_URL'),
            'timeout' => 5,
            //'connect_timeout' => 3,
            'allow_redirects' => false
        ]);

        $storeCode = $task->getStockUuid();

        $cachePathFinish = $this->getCachePathFinish($task->getId());

        if (file_exists($cachePathFinish)) {
            unlink($cachePathFinish);
        }

        $countAll = $moySkladService->getRemainCountByStock($storeCode);

        //получение кол-ва приемок
        if ($task->getPricelistSheetId() > 0) {
            $allSupplyCount = $countAll;
        } else {
            $allSupplyCount = $moySkladService->getSupplyCountByStock($storeCode);
        }


        $this->setData([], $task->getId(), $countAll, $allSupplyCount);




        //первый запрос на себя для загрузки пачками
        try {
            //первый запрос начиная от последней страницы, чтобы новые приемки были первыми
            $client->post('/_service/mysklad/remains/part/0', [
                'form_params' => [
                    'task' => $task->getId(),
                    'mark' => $mark
                ]
            ]);

        } catch (ConnectException $ex) {
            // $logger->debug($ex->getMessage());
        }

    }


    /**
     * @param int $task
     * @param bool $finish
     * @param bool $autoCreate
     * @return array
     */
    public function getData(int $task, bool $finish = false, bool $autoCreate = true): array
    {
        $path = !$finish ? $this->getCachePathProcess($task) :
            $this->getCachePathFinish($task);

        if (!file_exists($path) && $autoCreate === true) {
            file_put_contents($this->getCachePathProcess($task), json_encode([
                'results' => [],
                'countAll' => null,
                'supplyCount' => null
            ]));
        }
        elseif (!file_exists($path)) {
            return [];
        }

        if (!$result = file_get_contents($path)) {
            return [];
        }
        return json_decode($result, true);
    }

    /**
     * @param array $data
     * @param int $task
     * @param int|null $countAll
     * @param int|null $allSupplyCount
     */
    public function setData(array $data, int $task, int $countAll = null, int $allSupplyCount = null): void
    {
        $res = [];

        if (file_exists($this->getCachePathProcess($task))) {
            $res = $this->getData($task);
        }

        $res['results'] = $data;

        if ($countAll) {
            $res['countAll'] = $countAll;
        }

        if ($allSupplyCount) {
            $res['supplyCount'] = $allSupplyCount;
        }

        if (!isset($res['supplyCount'])) {
            $res['supplyCount'] = 0;
        }

        if (!isset($res['countAll'])) {
            $res['countAll'] = null;
        }

        file_put_contents($this->getCachePathProcess($task), json_encode($res));
    }

    /**
     * @param int $task
     * @param string|null $login
     * @param string|null $mark
     * @return string
     */
    public function createLoggerPath(int $task, ?string $login, ?string $mark): string
    {
        return MoySkladFactory::createLoggerPath($task, $login, $mark);
    }

    /**
     * @param MoySkladTask $task
     * @return string
     */
    public function getLastLog(MoySkladTask $task): string
    {
        if (!$task->getMark()) {
            return '';
        }
        return $this->getLastLogByFile(
            $this->createLoggerPath($task->getId(), $this->mainConfig->getMoyskladLogin(), $task->getMark())
        );
    }

    /**
     * @param string $file
     * @return string
     */
    public function getLastLogByFile($file = 'email.log'): string
    {

        if (!file_exists($file)) {
            return '';
        }

        $f = fopen($file, 'r');
        $list = '';
        if ($f) {
            if (fseek($f, -2, SEEK_END) === 0) {//в конец файла -2 символ перевода строки
                $len = ftell($f); //ftell — Сообщает текущую позицию чтения/записи файла

                for ($i = $len; $i > ($len - 1000); $i--) {//5000 - предполагаемая макс. длина строки

                    fseek($f, -2, SEEK_CUR);

                    fread($f, 1);
                }

                $res = fread($f, $len - $i);

                $list .= $res;//последняя строка

            }
            fclose($f);
        }

        $res = mb_convert_encoding($list, 'UTF-8', 'UTF-8');

        if (preg_match_all('~(\[.*?\])\s\[DEBUG\](.*?)\[Context~', $res, $match)) {

            $res = array_pop($match[1]) . ' ' . array_pop($match[2]);
        }

        return trim($res);
    }

    /**
     * @param MoySkladEntityInterface $task
     * @return TaskStatus
     */
    public function createTaskStatus(MoySkladEntityInterface $task): TaskStatus
    {

        $dir = BASEPATH . self::TASKS_FILES_DIR . $task->getId();

        if (!is_dir($dir) && !mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new MoySkladException('dir for work file not create ' . $dir);
        }

        $status = new TaskStatus();

        $processFile = $this->getCachePathProcess($task->getId());


        if (!$res = $this->getLastLog($task)) {
            $res = '';
        }
        $lastTime = time();

        if (preg_match('~\[(\d{2})\/(\d{2})\/(\d{4})\s(.*?)\]~is', $res, $match)) {
            $lastTime = date_create($match[3] . '-' . $match[2] . '-' . $match[1] . ' ' . $match[4])->getTimestamp();
        }

        //если больше 5 минут нет активности считаем зависшим
        if (file_exists($processFile) && (time() - $lastTime) / 60 > 10) {
            $statusName = 'Завис';
            $statusDisplay = '<span style=\'color: red\'>' . $statusName . '</span>';
            $code = self::STATUS_HOVERED_CODE;
        } elseif (file_exists($this->getCachePathFinish($task->getId()))) {
            $statusName = 'Завершен';
            $code = self::STATUS_FINISH_CODE;
            $statusDisplay = '<span style=\'color: #34A70E\'>' . $statusName . '</span>';
        } elseif (file_exists($processFile)) {
            $statusName = 'В работе';
            $statusDisplay = '<span style=\'color: blue\'>' . $statusName . '</span>';
            $code = self::STATUS_WORK_CODE;
        } else {
            $statusName = $statusDisplay = 'Ожидание';
            $code = self::STATUS_WAIT_CODE;
        }

        $status->setStatus($statusName);
        $status->setStatusDisplay($statusDisplay);
        $status->setCode($code);
        return $status;
    }


    /**
     * @param int $id
     * @return string
     */
    public function getCachePathFinish(int $id): string
    {
        $dir = BASEPATH . self::TASKS_FILES_DIR . $id . '/';
        return $dir . self::TASKS_FINISH_FILE;
    }

    /**
     * @param int $id
     * @return string
     */
    public function getCachePathProcess(int $id): string
    {
        $dir = BASEPATH . self::TASKS_FILES_DIR . $id . '/';
        return $dir . self::TASKS_PROCESS_FILE;
    }
}