<?php


namespace MoySklad\Application\Console;


use Local\Factory\Logger;
use MoySklad\Application\Commands\CreateOrder\CreateOrderCommand;
use MoySklad\Application\Commands\CreateOrder\CreateOrderException;
use MoySklad\Application\Commands\CreateOrder\CreateOrderWarningException;
use MoySklad\Application\MoySkladTaskService;
use MoySklad\Infrastructure\MoySkladTask\MoySkladTaskRepository;
use MoySklad\MoySklad;
use Shop\Config\Main;
use Shop\Models\Order\Order;
use Shop\Models\OrderSign\OrderSign;
use Shop\Repositories\CustomerOrder\CustomerOrderCriteria;
use Shop\Repositories\CustomerOrder\CustomerOrderRepository;
use Shop\Repositories\PricePosition\PricePositionRepository;
use spaceonfire\CommandBus\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class SyncOrdersMoySklad
 * @package MoySklad\Application\Console
 */
class SyncOrdersMoySklad extends Command
{
    protected $customerOrderRepository;
    protected $pricePositionRepository;
    protected $moySkladTaskService;

    protected $logger;

    protected $commandBus;

    protected $mainConfig;

    /**
     * SyncOrdersMoySklad constructor.
     * @param null $name
     * @param CommandBus $commandBus
     * @param CustomerOrderRepository $customerOrderRepository
     * @param MoySkladTaskService $moySkladTaskService
     * @param Main $mainConfig
     * @param PricePositionRepository $pricePositionRepository
     * @throws \Exception
     */
    public function __construct($name = null,
                                CommandBus $commandBus,
                                CustomerOrderRepository $customerOrderRepository,
                                MoySkladTaskService $moySkladTaskService,
                                Main $mainConfig,
                                PricePositionRepository $pricePositionRepository)
    {
        parent::__construct($name);
        $this->customerOrderRepository = $customerOrderRepository;
        $this->pricePositionRepository = $pricePositionRepository;
        $this->moySkladTaskService = $moySkladTaskService;

        $this->commandBus = $commandBus;
        $this->mainConfig = $mainConfig;
        $this->logger = Logger::buildFileLogger([\Monolog\Logger::INFO],'moysklad-orders-sync');
    }

    protected function configure()
    {
        $this->setName('tools:orders-sync')
            ->setDescription('Sync orders for moysklad');
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Propel\Runtime\Exception\PropelException
     */
    protected function execute(InputInterface $input, OutputInterface $output): void
    {

        $this->logger->debug('start sync orders');

        $stateId = $this->mainConfig->getMoyskladOrderStateId();
        $stateSucessId = $this->mainConfig->getMoyskladOrderStateSucessId();
        $stockIds = $this->mainConfig->getMoyskladSyncStockIds();
        $remoteStock = $this->mainConfig->getMoyskladRemoteStock();

        if(!$stateId || !$stateSucessId){

            $text = 'states for sync moysklad not set';

            $this->logger->error($text);

            $output->writeln($text);
            return;
        }

        $orders = $this->customerOrderRepository->findByCriteria(
            CustomerOrderCriteria::create()
                ->setFilterByStateId($stateId)
                ->setFilterByStockIds($stockIds)
        );

        $this->logger->debug('get orders, count');

        /**
         * @var Order $order
         */
        foreach ($orders as $order) {

            if(!$sign = $order->getOrderSign()){
                $sign = new OrderSign();
            }

            if(in_array($sign->getMoyskladSync(),['sucess','warning'])){

                $text = 'order ' .$order->getOrderCode(). ' already sign ' . $sign->getMoyskladSync();
                $this->logger->debug($text);
                $output->writeln($text);
                continue;
            }

            try{

                $command = new CreateOrderCommand($order, $remoteStock);

                $this->commandBus->handle($command);

                $order->setStateId($stateSucessId);

                //установить признак синхронизации как успешно
                $order->setOrderSign($sign->setMoyskladSync('sucess'));

                $text = 'create remote order id:' . $command->getResultId();

                $this->logger->debug($text);

                $output->writeln($text);
            }
            catch (CreateOrderWarningException $ex){

                $order->setOrderSign($sign->setMoyskladSync('warning'));

                $this->logger->error($ex->getMessage());

                $output->writeln($ex->getMessage());
            }

            catch (CreateOrderException $ex){

                //установить признак синхронизации как ошибка
                $order->setOrderSign($sign->setMoyskladSync('error'));

                $this->logger->error($ex->getMessage());

                $output->writeln($ex->getMessage());
            }


            $this->customerOrderRepository->save($order);
        }

        $text = 'sync orders completed!';
        $this->logger->debug($text);
        $output->writeln($text);
    }

}