<?php
/**
 * Created by PhpStorm.
 * User: Admin
 * Date: 20.05.2020
 * Time: 22:36
 */

namespace MoySklad\Infrastructure;


use GuzzleHttp\Exception\RequestException;
use MoySklad\Components\Http\MoySkladHttpClient;
use MoySklad\Domain\Provider\FullAddress;
use MoySklad\Domain\Provider\ProviderEntity;
use MoySklad\Domain\Remain\RemainCollection;
use MoySklad\Domain\Remain\RemainEntity;
use MoySklad\Domain\Stock\StockEntity;
use MoySklad\Entities\Products\Product;
use MoySklad\Models\MoySkladTask\MoySkladTaskQuery;
use MoySklad\MoySklad;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Symfony\Component\Filesystem\Tests\FilesystemTestCase;
use Twig\Cache\CacheInterface;

class MoySkladRepository implements LoggerAwareInterface
{
    use LoggerAwareTrait;
    use MoySkladClient;

    protected $cache;
    protected $login;
    protected $password;
    protected $cacheProviders = [];
    protected $cacheProducts = [];

    public function __construct(string $login, string $password, CacheInterface $cache)
    {
        $this->login = $login;
        $this->password = $password;
        $this->logger = new NullLogger();
        $this->cache = $cache;
    }

    /**
     * @param string $key
     * @param string $content
     */
    protected function addInCache(string $key, string $content)
    {
        $this->cache->write($this->cache->generateKey($key, $key), $content);
    }

    /**
     * @param string $key
     * @return string
     */
    protected function getInCache(string $key): ?string
    {
        ob_start();
        $this->cache->load($this->cache->generateKey($key, $key));

        return ob_get_clean();
    }


    public function getSupplyCountByStock(string $storeUuid): int
    {

        $httpClient = $this->getHttpClient();
        //Получение инфы о складе
        try {

            $key = 'entity/store/' . $storeUuid;

            if (!$result = $this->getInCache($key)) {
                $store = $httpClient->get('entity/store/' . $storeUuid);
                $this->addInCache($key, serialize($store));
            } else {
                $store = unserialize($result);
            }

        } catch (\GuzzleHttp\Exception\ServerException $ex) {
            $this->logger->error($ex->getMessage());
            throw new RepositoryException('Service Unavailable');
        }

        $storeLink = $store->meta->href;


        //получаем приемки
        try {

            $data = [
                'limit' => 1,
                'offset' => 0,
                'filter' => 'store=' . $storeLink
            ];

            $data = $httpClient->get('entity/supply', $data);

        } catch (\GuzzleHttp\Exception\ServerException $ex) {
            $this->logger->error($ex->getMessage());
            throw new RepositoryException('Service Unavailable');
        }

        return $data->meta->size;
    }


    /**
     * @param SearchCriteria $criteria
     * @return RemainCollection
     * @throws \Throwable
     */
    public function findByCriteria(SearchCriteria $criteria): RemainCollection
    {

        $httpClient = $this->getHttpClient();

        $remainCollection = new RemainCollection();

        $this->logger->debug(sprintf('получение приемок limit %s, offset %s', $criteria->getLimit(), $criteria->getPage()));

        $metaHref = null;

        if ($criteria->getFilterByContractorName() !== null) {
            $contragents = $httpClient->get('entity/counterparty', [
                'limit' => 1,
                'search' => $criteria->getFilterByContractorName() //может в будущем пригодиться фильтровать по имени документа приемки
            ]);
            foreach ($contragents->rows as $row) {
                $metaHref = $row->meta->href;
                break;
            }
        }

        //Получение инфы о складе
        try {

            $key = 'entity/store/' . $criteria->getFilterByStockId();

            if (!$result = $this->getInCache($key)) {
                $store = $httpClient->get('entity/store/' . $criteria->getFilterByStockId());
                $this->addInCache($key, serialize($store));
            } else {
                $store = unserialize($result);
            }

        } catch (\GuzzleHttp\Exception\ServerException $ex) {
            $this->logger->error($ex->getMessage());
            throw new RepositoryException('Service Unavailable');
        }

        $storeLink = $store->meta->href;


        //получаем приемки
        try {

            $data = [
                'limit' => $criteria->getLimit(),
                'offset' => $criteria->getPage(),
                'expand' => 'positions,agent',
                'filter' => 'store=' . $storeLink
            ];

            $this->logger->debug('запрос на получение приемок', $data);
            //dump($data);

//            if($metaHref!==null){
//                $data['filter=agent']=$metaHref;
//            }

           // $key = 'entity/supply_limit_' . md5(serialize($data));

//            if(!$result = $this->getInCache($key)){
//                $data = $httpClient->get('entity/supply', $data);
//                $this->addInCache($key,serialize($data));
//            }
//            else{
//                $data = unserialize($result);
//            }
            $data = $httpClient->get('entity/supply', $data);


        } catch (\GuzzleHttp\Exception\ServerException $ex) {
            $this->logger->error($ex->getMessage());
            throw new RepositoryException('Service Unavailable');
        }


        // dd($data);


        $supplyRows = $data->rows;

        $this->logger->debug('обработанно приемок ' . (count($supplyRows) + $criteria->getPage()) . ', всего ' . $data->meta->size);

        if (count($supplyRows) === 0) {
            $this->logger->debug('приемок больше нет, выходим');
            return $remainCollection;
        }

        //STOCK INFO
        //$this->logger->debug('получение информации о торг.точке, успешно');

        $stockEntity = (new StockEntity())->setName($store->name)->setId($store->id);

        //проход по примемках
        foreach ($supplyRows as $doc) {

            $textLog = 'приемка name: ' . $doc->name;

            if ($doc->applicable === false) {
                $this->logger->debug($textLog . ', приемка не активна, пропускаем');
                continue;
            }

            if (preg_match('~store\/(.*?)$~is', $doc->store->meta->href, $match)) {
                $stockId = $match[1];
            } else {
                $this->logger->debug($textLog . ', не нашли uuid склада у приемки');
                continue;
            }

            //если не наш склад сразу выходим
            if ($criteria->getFilterByStockId() !== $stockId) {
                $this->logger->debug($textLog . ', не наш склад, пропускаем, ' . $doc->store->meta->href);
                continue;
            }

            $this->logger->debug($textLog . ', наш склад, анализ позиций');

            try {
                $this->getSupplyPositions($remainCollection, $stockEntity, $httpClient, $doc, $criteria);
            } catch (PositionsSupplyNotFoundException $ex) {
                $this->logger->debug('позиций у приемки не найдено');
                continue;
            }


        }

        return $remainCollection;
    }

    /**
     * @param RemainCollection $remainCollection
     * @param StockEntity $stockEntity
     * @param $httpClient
     * @param $doc
     * @param SearchCriteria $criteria
     * @throws \Throwable
     */
    private function getSupplyPositions(RemainCollection $remainCollection, StockEntity $stockEntity, MoySkladHttpClient $httpClient, $doc,
                                        SearchCriteria $criteria): void
    {

        //кол-во позиций приемки
        $positionsCount = $doc->positions->meta->size;

        //if($doc->store->meta->href)
        if ($positionsCount === 0) {
            throw new PositionsSupplyNotFoundException('positions not found');
        }


        $contragent = $doc->agent;


        $provider = (new ProviderEntity)
            ->setName($contragent->name)
            ->setId($contragent->id);

        //если прилетел этот параметр то остальные не важны
        if (isset($contragent->actualAddressFull) && isset($contragent->actualAddressFull->addInfo)) {
            $provider->setAddress($contragent->actualAddressFull->addInfo);
        } else {
            if (isset($contragent->actualAddress)) {
                $provider->setAddress($contragent->actualAddress);
            }

            if (isset($contragent->legalAddressFull)) {
                $provider->setFullAddress(
                    new FullAddress(
                        $contragent->legalAddressFull->postalCode ?? null,
                        $contragent->legalAddressFull->city ?? null,
                        $contragent->legalAddressFull->street ?? null,
                        $contragent->legalAddressFull->house ?? null,
                        $contragent->legalAddressFull->apartment ?? null
                    )
                );
            }
        }

        $contragent = $provider;


        //контраген

        $this->logger->debug('получение информации о контрагенте, определен ' . $contragent->getName() . ' address ' . $contragent->getAddress());


        if ($criteria->getFilterByProviderAddress() && $contragent->getAddress() !== $criteria->getFilterByProviderAddress()) {
            $this->logger->debug('пропускаем контрагента, так как есть фильтр по направлению ' . $criteria->getFilterByProviderAddress());
            return;
        }

        // $offset = 0;
        // $limit = 10;

        //получаем позиции приемки ------------------


        $positions = $doc->positions;


        $foundCount = $positions->meta->size;

        $this->logger->debug(sprintf('всего позиций в приемке %s %s', $foundCount, ($foundCount > 100 ? 'ВНИМАНИЕ БОЛЬШЕ 100!!!!' : '')));

        if (!$foundCount) {
            $this->logger->debug('завершаем поиск позиций приемки');
            return;
        }

        //Проход по позициям приемки
        $offset = 0;
        $numPos = 0;
        $part = 0;
        $dbCheck = 0;

        //блок логики должен пройтись по всем позициям приемки от 0 до $foundCount
        while (true) {


            foreach ($positions->rows as $position) {

                $numPos++;
                $dbCheck++;

                if ($dbCheck > 50) {
                    MoySkladTaskQuery::create()->count();
                    $this->logger->debug('оживили базу');
                    $dbCheck = 0;
                }

                $this->logger->debug('анализ позиций приемки №' . ($offset + $numPos));

                $productLink = $position->assortment->meta->href;

                if (preg_match('~product\/(.*?)$~is', $productLink, $match)) {
                    $uuid = $match[1];
                } else {
                    throw new RepositoryException('uuid product not found');
                }

                //получение остатков по товару
                try {

                    $data = [
                        'limit' => 100,
                        'store.id' => $criteria->getFilterByStockId(),
                        'product.id' => $uuid,
                        'stockMode' => 'positiveOnly',
                        'groupBy' => 'product'
                    ];

                    $remains = $httpClient->get('report/stock/all', $data);


                } catch (\GuzzleHttp\Exception\ServerException $ex) {
                    $this->logger->error($ex->getMessage());
                    throw new RepositoryException('Service Unavailable');
                }

                $this->logger->debug(
                    'получение остатков для позиции приемки, найдено ' . count($remains->rows) . ' всего ' . $remains->meta->size);

                if (count($remains->rows) === 0) {
                    $this->logger->debug('остатков нет, пропускаем');
                    continue;
                }

                //проход по остаткам
                foreach ($remains->rows as $remain) {

                    $this->buildRemainPosition(
                        $remainCollection,
                        $remain,
                        $criteria,
                        $httpClient,
                        $productLink,
                        $contragent,
                        $stockEntity
                    );
                }

            }

            //идем на след круг
            if($numPos < $foundCount){
                $part++;
                $positions = $httpClient->get($doc->positions->meta->href, [
                    'limit' => 100,
                    'offset' => $part * 100
                ]);
            }
            else{
                //выход из цикла
                break;
            }

        }


    }

    /**
     * @param RemainCollection $remainCollection
     * @param \stdClass $remain
     * @param SearchCriteria $criteria
     * @param MoySkladHttpClient $httpClient
     * @param string $productLink
     * @param ProviderEntity $provider
     * @param StockEntity $stockEntity
     * @return void
     * @throws \Repo\Concrete\Exceptions\Collection
     * @throws \Throwable
     */
    protected function buildRemainPosition(RemainCollection $remainCollection,\stdClass $remain, SearchCriteria $criteria,
                                           MoySkladHttpClient $httpClient, string $productLink,ProviderEntity $provider,
                                           StockEntity $stockEntity
    ):void {

        //$key = $productLink;

        $product = $this->getProductWithCache($productLink, $httpClient);
//        if (!$result = $this->getInCache($key)) {
//
//            $this->addInCache($key, serialize($product));
//        } else {
//            $product = unserialize($result);
//        }

        if ($criteria->getFilterBySalePrice() > '') {

            foreach ($product->salePrices as $salePriceOb) {
                if ($salePriceOb->priceType === $criteria->getFilterBySalePrice()) {
                    $finalPrice = $salePriceOb->value;
                    break;
                }
            }
        } else {
            $finalPrice = $remain->price;
        }

        //нулевые цены нам не нужны и остатки
        if (!isset($finalPrice) || !$remain->stock || !$remain->name) {

            $this->logger->debug(
                'пропускаем позицию, нет обязательных параметров ,external code: ' .
                $remain->externalCode
            );

            return;
        }

        $code = $remain->code??$product->code;

        if(!$code){
            $this->logger->debug(
                'пропускаем позицию, нет обязательных кода позиции, ,external code: ' .
                $remain->externalCode
            );
            return;
        }

        $key = $remain->externalCode;//md5($finalPrice.$remain->stock.$remain->name.$remain->externalCode);

        $remainEntity
            = (new RemainEntity())
            ->setRemains($remain->stock)
            ->setName($remain->name)
            ->setCode($code)
            ->setPrice($finalPrice)
            ->setProvider($provider)
            ->setStock($stockEntity)
            ->setWeight($product->weight)
            ->setId($key);



        $this->logger->debug('определили остаток, добавляем в коллекцию: ' . $remainEntity->getId() .
            ' name: '. $remainEntity->getName() );

        $remainCollection->push($remainEntity);

    }


    /**
     * @param string $productLink
     * @param MoySkladHttpClient $httpClient
     * @return \stdClass
     * @throws \Throwable
     */
    public function getProductWithCache(string $productLink, MoySkladHttpClient $httpClient): \stdClass
    {
        $key = md5($productLink);

        if (isset($this->cacheProducts[$key])) {
            return $this->cacheProducts[$key];
        }

        if (!preg_match('~product/(.*?)$~is', $productLink, $match)) {
            throw new RepositoryException('not parse product uuid from ' . $productLink);
        }

        $uuid = $match[1];
        // $connection = MoySklad::getInstance($this->login, $this->password);

        $loop = 0;

        $product = null;

        //пробуем читать инфу несколько интерации
        while ($loop < 10) {
            try {

                $product = $httpClient->get('entity/product/' . $uuid);

                break;
            } catch (RequestException $ex) {
                $loop++;
                sleep(1);
            }
        }

        if (!$product) {
            throw new RepositoryException('not get product from moysklad by uuid' . $uuid);
        }

        $this->cacheProducts[$key] = $product;

        return $this->cacheProducts[$key];
    }

    /**
     * @param string $link
     * @param MoySkladHttpClient $httpClient
     * @return ProviderEntity
     * @throws \Throwable
     */
    public function getContragentWithCache(string $link, MoySkladHttpClient $httpClient): ProviderEntity
    {
        $key = md5($link);

        if (isset($this->cacheProviders[$key])) {
            return $this->cacheProviders[$key];
        }

        $contragent = $httpClient->get($link);

        $provider = (new ProviderEntity)
            ->setName($contragent->name)
            ->setId($contragent->id);

        //если прилетел этот параметр то остальные не важны
        if (isset($contragent->actualAddressFull) && isset($contragent->actualAddressFull->addInfo)) {
            $provider->setAddress($contragent->actualAddressFull->addInfo);
        } else {
            if (isset($contragent->actualAddress)) {
                $provider->setAddress($contragent->actualAddress);
            }

            if (isset($contragent->legalAddressFull)) {
                $provider->setFullAddress(
                    new FullAddress(
                        $contragent->legalAddressFull->postalCode ?? null,
                        $contragent->legalAddressFull->city ?? null,
                        $contragent->legalAddressFull->street ?? null,
                        $contragent->legalAddressFull->house ?? null,
                        $contragent->legalAddressFull->apartment ?? null
                    )
                );
            }
        }

        $this->cacheProviders[$key] = $provider;

        return $provider;
    }

}