<?php

namespace MoySklad\Application\Controllers;

use Core\BaseController;
use Core\Concrete\RequestAwareTrait;
use Core\Contracts\RequestAwareInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use Local\Config\Main;

use MoySklad\Application\Commands\UpdatePriceListFromRemains\UpdatePriceListsByTaskCommand;
use MoySklad\Application\Exceptions\MoySkladException;
use MoySklad\Application\MoySkladFactory;
use MoySklad\Application\MoySkladTaskService;
use MoySklad\Infrastructure\MoySkladTask\MoySkladTaskRepository;
use MoySklad\Models\MoySkladTask\MoySkladTask;
use MoySklad\MoySklad;
use Psr\Log\LoggerInterface;
use Shop\Repositories\Brand;
use Shop\Repositories\ProductCategory\ProductCategoryCriteria;
use Shop\Repositories\ProductCategory\ProductCategoryRepository;
use spaceonfire\CommandBus\CommandBus;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;


/**
 * Description of Mysklad
 *
 * @author Admin
 */
class Mysklad extends BaseController implements RequestAwareInterface
{

    use RequestAwareTrait;

    //кол-во приемок за интерацию
    protected $stepLimit = 20;

    protected $session;
    protected $priceListScheet;

    protected $loggerSync;

    protected $control;

    protected $brandRepo;
    protected $categoryRepo;
    protected $baseUrl;

    /**
     * принудительный лимит для отладки
     * @var int|null
     */
    protected $forceLimit;

    protected $moySkladTaskService;
    protected $moySkladTaskRepository;
    protected $commandBus;
    protected $mainConfig;


    public function __construct(
        \Symfony\Component\HttpFoundation\Session\Session $session,
        Brand\BrandRepository $brandRepo,
        \Shop\Services\PriceListScheet $priceListScheet,
        \Core\Form\ControlBuilder $control,
        ProductCategoryRepository $categoryRepo,
        MoySkladTaskService $moySkladTaskService,
        MoySkladTaskRepository $moySkladTaskRepository,
        CommandBus $commandBus,
        Main $mainConfig
    )
    {
        $this->session = $session;
        $this->control = $control;
        $this->brandRepo = $brandRepo;
        $this->priceListScheet = $priceListScheet;
        $this->categoryRepo = $categoryRepo;
        $this->moySkladTaskService = $moySkladTaskService;
        $this->moySkladTaskRepository = $moySkladTaskRepository;
        $this->mainConfig = $mainConfig;
        $this->commandBus = $commandBus;
    }

    /**
     * @return Client
     * @throws \Exception
     */
    private function getHttpClient(): Client
    {
        for ($i = 0; $i < 10; $i++) {
            $this->baseUrl = getenv('BASE_URL');
            if (!is_string($this->baseUrl)) {
                sleep(3);
//                $dotenv = Dotenv::create(BASEPATH);
//                $dotenv->load();

                $this->logger->debug('проблема с base_url `%s`, пробуем переподключить env ' . $this->baseUrl);

            } else {
                break;
            }
        }

        try {
            $fp = fopen('php://temp', 'r+');
            $client = new Client([
                'base_uri' => $this->baseUrl,
                'timeout' => 3,
                'connect_timeout' => 3,
                'debug' => $fp,
                'allow_redirects' => false
            ]);
            fseek($fp, 0);
            return $client;
        } catch (\Exception $ex) {
            $this->logger->error(stream_get_contents($fp));
            throw $ex;
        }

    }

    /**
     * получение контрола складов
     * @return Response
     * @throws \Throwable
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function getStocksDropDown(): Response
    {

        $login = $this->request->query->get('login');
        $pass = $this->request->query->get('pass');

        if (!$login || !$pass) {
            return new Response('', Response::HTTP_BAD_REQUEST);
        }

        $moySkladService = MoySkladFactory::buildService($login,$pass);

        $stocks = $moySkladService->getStosks();

        $priceListControl = $this
            ->control
            ->buildDropdownUniform('stock_id', ['-не выбранно-'] + $stocks->getDropdownList())
            ->setClass('required');

        $template = $this->twig->loadTemplate('@moysklad/form.twig');

        return new Response(
            $template->render([
                'stocksControl' => $priceListControl->render()
            ])
        );
    }

    /**
     * @param array $data
     * @param int $task
     * @param int|null $countAll
     * @param int|null $allSupplyCount
     */
    private function setData(array $data, int $task, int $countAll = null, int $allSupplyCount = null): void
    {
        $this->moySkladTaskService->setData($data, $task, $countAll, $allSupplyCount);
    }

    /**
     * @param int $task
     * @param bool $finish
     * @return array
     */
    private function getData(int $task, bool $finish = false): array
    {
        return $this->moySkladTaskService->getData($task,$finish);
    }

    /**
     * @param string $file
     * @return string
     */
    private function getLastLog($file = 'email.log'): string
    {
        return $this->moySkladTaskService->getLastLogByFile($file);
    }

    /**
     * статус процеса загрузки
     * @return Response
     * @throws \Exception
     */
    public function getRemainStatus(): Response
    {
        //-----------------------------------------------------------------------
        $params = $this->request->query->all();

        $mark = (int)$params['mark'];
        $task = (int)$params['task'];

        if (!$taskEntity = $this->moySkladTaskRepository->findById($task)) {
            throw new MoySkladException('taskEntity not found');
        }

        $login = $this->mainConfig->getMoyskladLogin();

        //получаем сервис
        $service = MoySkladFactory::buildService('login', '123', $this->logger);

        //-----------------------------------------------------------------------
        if (file_exists($this->moySkladTaskService->getCachePathProcess($task))) {
            $data = $this->getData($task);
            $file = $this->createLoggerPath($task, $login, $mark);
            return new JsonResponse([
                'status' => 'process',
                'log' => $this->getLastLog($file),
                'count' => count($data['results']),
                'countAll' => $data['countAll'],
                'supplyCount' => $data['supplyCount']
            ], Response::HTTP_OK);
        }

        //-----------------------------------------------------------------------
        if (!file_exists($this->moySkladTaskService->getCachePathFinish($task))) {
            $taskEntity->setMark(null);
            $taskEntity->save();
            return new JsonResponse([
                'status' => 'process'
            ], Response::HTTP_OK);
        }

        //-----------------------------------------------------------------------
        //отображаем данные
        $result = $this->getData($task, true);
        $data = $result['results'];

        $newProducts = [];
        $i = 0;
        $newData = [];

        /**
         * @var MoySkladTask $taskEntity
         */
        $setPriceId = $taskEntity->getPricelistSheetId();

        //привязывваем прайсы
        foreach ($data as $item) {

            if (!$setPriceId) {
                $priceListId = $service->getPriceListId(0, $item['provider']);
            } else {
                $priceListId = $setPriceId;
            }


            $item['priceListId'] = $priceListId;

            if ((int)$priceListId === 0) {
                $item['status'] = '';
            }

            $newData[$i] = $item;
            $newData[$i]['id'] = $newData[$i]['i'] = $i + 1;

            if ($item['new'] === true) {
                $newProducts[] = $newData[$i]['i'];
            }

            $i++;
        }

        //Автопроиск прайсов
        $pricesData = $service->prepareArrayRemainsDataGroupPrices($newData);
        $data = [];
        /**
         * @var MoySkladTask $taskEntity
         */
        foreach ($pricesData as $priceId => $remains) {

            $items = $service->buildPrice($remains, (int)$priceId, $taskEntity->getCategoryId(), $taskEntity->getBrandId(), $this->logger, true);
            $data += $items;
        }

        ksort($data);


        $list = '';

        $brandsCollection = $this->brandRepo->findByCriteria(
            Brand\BrandCriteria::create()->setSortByName('asc')
        );
        foreach ($brandsCollection as $brand) {
            $list .= sprintf('%s:%s;', $brand->getId(), $brand->getName());
        }

        $list2 = '';

        $categoriesCollection = $this->categoryRepo->findByCriteria(
            ProductCategoryCriteria::create()->setSortByTitle('asc')
        );

        foreach ($categoriesCollection as $category) {
            /**
             * @var \Shop\Models\Category\Category $category
             */
            $list2 .= sprintf('%s:%s;', $category->getId(), $category->getNameWithParentName());
        }

        return new JsonResponse([
            'status' => 'finish',
            'count' => count($newData),
            'items' => $newData,
            'brands' => trim($list, ';'),
            'categories' => trim($list2, ';'),
            'newProducts' => $newProducts
        ], Response::HTTP_OK);

    }


    /**
     * подключение к скалду и загрузка остатков
     * @return JsonResponse|Response
     * @throws \Throwable
     */
    public function getRemains()
    {

        $taskId = (int)$this->request->query->get('task');
        /**
         * @var MoySkladTask $taskEntity
         */
        if (!$taskEntity = $this->moySkladTaskRepository->findById($taskId)) {
            throw new MoySkladException(sprintf('task by id `%s` not found', $taskId));
        }

        $login = $this->mainConfig->getMoyskladLogin();

        $pass = $this->mainConfig->getMoyskladPassword();

        $categoryDef = $taskEntity->getCategoryId();
        $brandDef = $taskEntity->getBrandId();



        $cachePath = $this->moySkladTaskService->getCachePathProcess($taskId);

        if (file_exists($cachePath)) {
            return new JsonResponse([
                'result' => 'process'
            ], Response::HTTP_OK);
        }

        if (!$categoryDef || !$brandDef || !$login || !$pass) {
            return new Response('validation error', Response::HTTP_BAD_REQUEST);
        }

        $mark = time();

        $logger = $this->buildLogger($login, $mark, $taskId);


        $logger->debug('подключение к моему складу, пользователь ' . $login . ' - ' . $pass);


        $moySkladService = MoySkladFactory::buildService($login, $pass, $logger);


        $this->moySkladTaskService->loadMoySkladTaskFirst($taskEntity, $mark, $moySkladService);

        return new JsonResponse([
            'id' => $mark
        ], Response::HTTP_OK);
    }


    public function test()
    {
        $login = $this->request->request->get('login');
        $mark = (int)$this->request->request->get('mark');
        $task = (int)$this->request->request->get('task');
        $pass = $this->request->request->get('pass');
        $storeCode = (string)$this->request->request->get('store');
        $salePrice = (string)$this->request->request->get('salePrice');

        //-----------
        $connection = MoySklad::getInstance($login, $pass);

        $loop = 0;
        while ($loop < 3) {
            try {
                $product = \MoySklad\Entities\Products\Product::query(
                    $connection
                )->byId('4458e5a0-69d5-11ea-0a80-059300264eb5');

                dump($product);

                break;
            } catch (RequestException $ex) {
                $loop++;
                sleep(15);
            }
        }


//        $logger = $this->buildLogger($login, $mark, $task);
//        $moySkladService = MoySkladFactory::buildService($login, $pass, $logger);
//
//        $remainsCollection = $moySkladService->getStockRemainsByStockId($storeCode, $salePrice, 0,10);
//
//        dump($remainsCollection);

        exit("test");
    }

    /**
     * /part/*
     * @param array $args
     * @return Response
     * @throws \Propel\Runtime\Exception\PropelException
     * @throws \Throwable
     */
    public function getRemainsPart(array $args): Response
    {
        /**
         * @var MoySkladTask $taskEntity
         */
        $taskId = (int)$this->request->request->get('task');
        if (!$taskEntity = $this->moySkladTaskRepository->findById($taskId)) {
            throw new \RuntimeException('taskEntity not found by Id ' . $taskId);
        }

        @set_time_limit(0);
        @ini_set('max_execution_time', 0);
        @ini_set('max_input_time', '3600');
        @ini_set('upload_max_filesize', '20M');
        @ini_set('post_max_size', '20M');
        ignore_user_abort(true);

        $part = (int)$args['part'];
        $mark = (int)$this->request->request->get('mark');


        $forceEnd = false;

        $login = $this->mainConfig->getMoyskladLogin();
        $pass = $this->mainConfig->getMoyskladPassword();

        $priceId = (int)$taskEntity->getPricelistSheetId();
        $storeCode = $taskEntity->getStockUuid();
        $categoryDef = $taskEntity->getCategoryId();
        $brandDef = $taskEntity->getBrandId();
        $salePrice = $taskEntity->getSalePrice();

        if (!$categoryDef || !$brandDef || !$login || !$pass) {
            return new Response('validation error', Response::HTTP_BAD_REQUEST);
        }

        $step = $this->stepLimit;

        //если приход по приемкам то лимит 1
        if (!$priceId) {
            $step = 1;
        }

        //если на предыдущем шаге не сформировался создадим новый
        $logger = $this->buildLogger($login, $mark, $taskId);

        $logger->debug('################################################### getRemainsPart ' . $part);

        //$price = $this->findPriceListById($priceId);
        $result = $this->getData($taskId);

        $data = $result['results'];
        $countAll = $result['countAll'];
        $supplyCount = $result['supplyCount'];

        $moySkladService = MoySkladFactory::buildService($login, $pass, $logger);

        if (!$countAll) {
            $countAll = $moySkladService->getRemainCountByStock($storeCode);
        }


        //используем получение остатков через проводки
        if ($priceId > 0) {

            $logger->debug(sprintf('поиск остатков по конкретному складу от %s до %s (всего %s)', $part * $step, $part * $step + $step, $supplyCount));

            $remainsCollection = $moySkladService->getStockRemainsBySupplyStockId(
                $storeCode,
                null,
                $salePrice,
                $part * $step,
                $step
            );

        } //используем получение остатков напрямую
        else {

            $logger->debug(sprintf('поиск остатков приемок от %s до %s (всего %s)', $part * $step, $part * $step + $step, $supplyCount));

            $remainsCollection = $moySkladService->getStockRemainsByStockId(
                $storeCode,
                null,
                $salePrice,
                $part * $step,
                $step
            );
        }

        $logger->debug('найдено остатков ' . $remainsCollection->count());

        //коллекция в массив
        if ($part * $step > $supplyCount && $remainsCollection->count() === 0) {
            $forceEnd = true;
        }

        $part++;

        $logger->debug('привязка товаров');

        //priceId - int или 0
        $newData = $moySkladService->convertRemainCollectionToArray($remainsCollection, $priceId);

        $endCount = count($newData);

        $logger->debug('convertRemainCollectionToArray - success, ' . $endCount .
            ' разница остатков ' . ($remainsCollection->count() - $endCount));

        $pricesData = $moySkladService->prepareArrayRemainsDataGroupPrices($newData);


        foreach ($pricesData as $priceId => $remains) {

            $logger->debug('walk $pricesData , остатков ' . count($remains));

            $items = $moySkladService->buildPrice($newData, (int)$priceId, $categoryDef, $brandDef, $logger, true);
            $data += $items;
        }

        $logger->debug('успешно, всего товаров ' . count($data) . ' из ' . $countAll);

        $this->setData($data, $taskId, $countAll);


        if ($forceEnd === true || ($this->forceLimit > 0 && count($data) > $this->forceLimit)) {//count($data) >= $countAll
            rename($this->moySkladTaskService->getCachePathProcess($taskId), $this->moySkladTaskService->getCachePathFinish($taskId));

            $logger->debug('завершаем, формируем конечный файл finish, кол-во ' . count($data), $taskEntity->toArray());

            /**
             * @var MoySkladTask $taskEntity
             */
            $taskEntity->setDateTimeEnd(new \DateTime());
            $taskEntity->setComplete(0);
            $taskEntity->save();

            $logger->debug('обновили task ', $taskEntity->toArray());

            return new Response('success, finish', Response::HTTP_OK);
        }

        $logger->debug('делаем запрос ' . '/_service/mysklad/remains/part/' . $part);

        try {

            $taskEntity->setComplete($part);
            $taskEntity->save();

            $this->getHttpClient()->post('/_service/mysklad/remains/part/' . $part, [
                    'form_params' => $this->request->request->all()
                ]
            );
        } catch (ConnectException $ex) {
            //$logger->debug($ex->getMessage());
        }


        return new Response('success, run next part ' . $part, Response::HTTP_OK);
    }




    /**
     * @param string $login
     * @param int $mark
     * @param int $task
     * @return LoggerInterface
     * @throws \Exception
     */
    private function buildLogger(string $login, int $mark, int $task): LoggerInterface
    {
        return MoySkladFactory::buildLogger( $login,$mark,$task);
    }

    /**
     * @param int $task
     * @param string $login
     * @param string $mark
     * @return string
     */
    private function createLoggerPath(int $task, string $login, string $mark): string
    {
        return MoySkladFactory::createLoggerPath($task, $login, $mark);
    }




    /**
     * @return JsonResponse
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function commitRemains()
    {
        $params = $this->request->request->all();
        $taskId = (int)$params['task'];
        $ids = (array)$params['ids'];

        $command = new UpdatePriceListsByTaskCommand($taskId,$ids);

        $this->commandBus->handle($command);

        return new JsonResponse([
            'result' => sprintf('прайс-лист успешно сохранен, созданно %s записей', $command->getEffectedRows())
        ], Response::HTTP_OK);
    }


}