<?php

namespace MoySklad\Application;

use MoySklad\Domain\Remain\RemainCollection;
use MoySklad\Domain\Remain\RemainEntity;
use MoySklad\Domain\Stock\StockCollection;
use MoySklad\Infrastructure\MoySkladRepository;
use MoySklad\Infrastructure\SearchCriteria;
use MoySklad\Infrastructure\StockRepository;
use Psr\Log\LoggerInterface;
use Shop\Models\PricelistSheet\PricelistSheetQuery;
use Shop\Models\Product\Product;
use Shop\Repositories\Brand\BrandCriteria;
use Shop\Repositories\Brand\BrandRepository;

class MoySkladService
{
    protected $moySkladRepository;
    protected $stockRepository;
    protected $moySkladSupplyRepository;
    protected $brandRepo;
    private $cachePrices = [];
    private $casheBrands = [];
    protected $cacheProducts = [];
    protected $cache = [];


    /**
     * MoySkladService constructor.
     * @param MoySkladRepository $moySkladRepository
     * @param BrandRepository $brandRepo
     * @param StockRepository $stockRepository
     * @param $moySkladSupplyRepository
     */
    public function __construct(MoySkladRepository $moySkladRepository,
                                BrandRepository $brandRepo,
                                StockRepository $stockRepository,
                                $moySkladSupplyRepository)
    {
        $this->moySkladRepository = $moySkladRepository;
        $this->stockRepository = $stockRepository;
        $this->moySkladSupplyRepository = $moySkladSupplyRepository;
        $this->brandRepo = $brandRepo;
    }

    /**
     * @param string $stockId
     * @param string|null $contractor
     * @param string|null $salePrice
     * @param int $offset
     * @param int $limit
     * @return RemainCollection
     * @throws \Throwable
     */
    public function getStockRemainsByStockId(string $stockId, ?string $contractor, ?string $salePrice, int $offset, int $limit): RemainCollection
    {
        $criteria = SearchCriteria::create()
            ->setFilterByStockId($stockId)
            ->setFilterBySalePrice($salePrice)
            ->setFilterByContractorName($contractor)
            //->setFilterByProviderAddress($destination)
            ->setLimit($limit)
            ->setPage($offset);

        return $this->moySkladRepository->findByCriteria($criteria);
    }

    /**
     * @param string $stockId
     * @param string|null $contractor
     * @param string|null $salePrice
     * @param int $offset
     * @param int $limit
     * @return RemainCollection
     */
    public function getStockRemainsBySupplyStockId(string $stockId, ?string $contractor, ?string $salePrice, int $offset, int $limit): RemainCollection
    {
        $criteria = SearchCriteria::create()
            ->setFilterByStockId($stockId)
            ->setFilterBySalePrice($salePrice)
            ->setFilterByContractorName($contractor)
            //->setFilterByProviderAddress($destination)
            ->setLimit($limit)
            ->setPage($offset);

        return $this->moySkladSupplyRepository->findByCriteria($criteria);
    }

    /**
     * @param string $stockuuId
     * @return int
     */
    public function getSupplyCountByStock(string $stockuuId): int
    {
        return $this->moySkladRepository->getSupplyCountByStock($stockuuId);
    }


    /**
     * @return StockCollection
     * @throws \Repo\Concrete\Exceptions\Collection
     * @throws \Throwable
     */
    public function getStosks(): StockCollection
    {

        return $this->stockRepository->findByCriteria(
            SearchCriteria::create()
        );
    }

    /**
     * @param string $stockId
     * @return int
     * @throws \Throwable
     */
    public function getRemainCountByStock(string $stockId): int
    {
        return $this->stockRepository->getRemainCountByStock($stockId);
    }


    /**
     * Приобразование данных в массив для сохранения в дампе
     * @param RemainCollection $remainsCollection
     * @param int $priceId
     * @return array
     */
    public function convertRemainCollectionToArray(RemainCollection $remainsCollection, int $priceId): array
    {
        $newData = [];

        /**
         * @var RemainEntity $remain
         */
        foreach ($remainsCollection as $remain) {


            $id = $remain->getId();

            $priceListId = $this->getPriceListId($priceId, $remain->getProvider()->getName());

//            if(!$priceListId){
//                continue;
//            }
//            if (isset($newData[$id])) {
//                $logger->debug('товар уже есть', $newData[$id]);
//                continue;
//            }

            $newData [$id] = [
                'id' => $id,
                'name' => $remain->getName(),
                'code' => $remain->getCode(),
                'price' => $remain->getPrice() / 100,// позиции прилиетают в 100 кратгном увеличении,@todo разобраться
                'currency' => $remain->getCurrency(),
                'stock' => $remain->getRemains(),
                'weight' => $remain->getWeight(),
                'priceListId' => $priceListId,
                'provider' => $remain->getProvider()->getName(),
                'provider_city' => $remain->getProvider()->getFullAddress() ?
                    $remain->getProvider()->getFullAddress()->getCity() : $remain->getProvider()->getAddress()
            ];


        }

        return $newData;
    }

    /**
     * сохранение массива с остатками
     * @param array $data
     * @return array
     */
    public function prepareArrayRemainsDataGroupPrices(array $data): array
    {

        $prices = [];

        foreach ($data as $key => $row) {
            $prices[(int)$row['priceListId']][$key] = $row;
        }

        return $prices;
    }

    /**
     * Получение id прайса поставщика в ИМ по прайслисту или поиску по контрагенту
     * @param int $priceId
     * @param string $contractorName
     * @return int|null
     */
    public function getPriceListId(int $priceId, string $contractorName): ?int
    {
        $key = $priceId . $contractorName;

        if ($priceId > 0) {
            return $priceId;
        }

        if (isset($this->cachePrices[$key])) {
            return $this->cachePrices[$key];
        }

        $priceListQ = PricelistSheetQuery::create()
            ->usePricelistFileQuery()
            ->useProviderQuery()
            ->useContractorQuery()
            ->filterByName($contractorName)
            ->endUse()
            ->endUse()
            ->endUse();

        $priceList = $priceListQ->findOne();


//        if(!$priceList){
//            dd($priceList,$contractorName,$priceListQ->toString());
//        }

        $this->cachePrices[$key] = $priceList ? $priceList->getId() : null;

        return $this->cachePrices[$key];
    }

    /**
     * @param int $priceId
     * @return \Shop\Models\PricelistSheet\PricelistSheet
     */
    protected function findPriceListById(int $priceId)
    {

        return \Shop\Models\PricelistSheet\PricelistSheetQuery::create()
            ->findOneById($priceId);
    }



    /**
     * @param string $article
     * @param string $name
     * @return Product|null
     */
    private function findByArticleOrName(string $article, string $name): ?Product
    {

        //$article = 'dsahjhvhj';
        // $name = 'Табак кальянный, ICE FRUIT GUM HL, 100 гр, СПЕКТРУМ';

        //$name = preg_replace('~[^A-ZА-Я0-9]~uis', '', mb_strtolower($name));
        $key = md5($article . $name);


        if (isset($this->cache[$key])) {
            return $this->cache[$key];
        }

        $product = \Shop\Models\Product\ProductQuery::create()
            ->findOneByArticle($article);

        if (!$product) {
            $product = \Shop\Models\Product\ProductQuery::create()
                ->findOneByTitle($name);
        }

        //dump($name, $product);
        //exit('dhjf');

        if (!$product) {
            return null;
        }

        $this->cache[$key] = $product;

        return $this->cache[$key];
    }

    protected function getBrandNames(): array
    {
        $items = [];

        if (count($this->casheBrands)) {
            return $this->casheBrands;
        }

        foreach ($this->brandRepo->findByCriteria(BrandCriteria::create()) as $brand) {
            /**
             * @var \Shop\Models\Brand\Brand $brand
             */
            $items [$brand->getId()] = $brand->getName();
        }

        $this->casheBrands = $items;

        return $items;
    }


    /**
     * @param LoggerInterface $logger
     * @param string $title
     * @param string $article
     * @param int $categoryDef
     * @param int $brandDef
     * @return Product|null
     */
    protected function autoCreateProduct(LoggerInterface $logger,
                                         string $title,
                                         string $article,
                                         int $categoryDef,
                                         int $brandDef): ?Product
    {
        $findBrandId = null;

        $logger->debug('find Brand brand names');

        foreach ($this->getBrandNames() as $brandId => $brandName) {

            if (mb_strripos($title, $brandName) !== false) {
                $findBrandId = $brandId;
                break;
            }
        }

        $logger->debug('findBrand id' . $findBrandId);

        if (!$findBrandId) {
            $findBrandId = $brandDef;
        }

        $categoryId = $categoryDef;

        $product = (new Product())
            ->setArticle($article)
            ->setCategoryId($categoryId)
            ->setBrandId($findBrandId)
            ->setTitle($title);

        return $product;
    }


    /**
     * @param array $remains
     * @param int $priceId
     * @param int|null $categoryDef
     * @param int|null $brandDef
     * @param LoggerInterface $logger
     * @param bool $buildItems
     * @param array|null $filterIds
     * @return array|\Shop\Models\PricelistSheet\PricelistSheet
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function buildPrice(array $remains,
                                  int $priceId,
                                  ?int $categoryDef,
                                  ?int $brandDef,
                                  LoggerInterface $logger,
                                  $buildItems = false,
                                  ? array $filterIds = null
    )
    {

        if (is_array($filterIds)) {
            array_walk($filterIds, function (&$value) {
                $value = (int)$value;
            });
        }


        if ($buildItems === false && !$price = $this->findPriceListById($priceId)) {
            throw new \RuntimeException('price not found by Id ' . $priceId);
        }

        if ($buildItems === false && $price) {
            $price->getPositions()->clear();
        }

        $items = [];
        $i = 0;
        foreach ($remains as $k => $row2) {
            $i++;
            $row = $row2;

            if (is_array($filterIds) && count($filterIds) && !in_array($i, $filterIds, true)) {
                $logger->error(sprintf('применение фильтра, позиция %s игнорируется', $i), $row);
                continue;
            }

            if (empty($row['code'])) {
                $logger->error(sprintf('У позиции %s не указан артикул/код', $row['code']), $row);
                continue;
            }

            $row['status'] = 'готов к загрузке';
            $row['new'] = false;

            $logger->debug('find product success by code or name ' . $row['code'] . ' ' . $row['name']);

            /**
             * @var Product $product
             */
            if (!$product = $this->findByArticleOrName((string)$row['code'], (string)$row['name'])) {

                $newName = $row['name'];
                $newArticle = $row['code'];
                $newCategory = $row['newCategory'] ?? ($categoryDef ?? $row['category']);
                $newBrand = $row['newBrand'] ?? ($brandDef ?? $row['brand']);

                $logger->debug('not found, create product', [
                    $newName,
                    $newArticle,
                    $newCategory,
                    $newBrand
                ]);

                //dump($row);

                if (!$product = $this->autoCreateProduct(
                    $logger,
                    $newName,
                    $newArticle,
                    $newCategory,
                    $newBrand
                )) {
                    $logger->debug('ошибка создания');
                    $row['status'] = "<span class='alert-product'>ошибка создания</span>";
                } else {

                    $logger->debug('новый товар');

                    $row['new'] = true;
                    $row['brand'] = $product->getBrandId();
                    $row['category'] = $product->getCategoryId();


                    //dump($product);
                    $product->setWeight($row['weight']);
                    $product->setdateUpdate(new \DateTime());

                    if ($buildItems === false) {
                        $price->addPosition($this->createPosition($row, $product));
                    }

                    $row['status'] = "<span class='new-product'>новый товар</span>";

                }

            } else {

                $logger->debug('found №' . $product->getId());

                $product->setWeight($row['weight']);
                $product->setdateUpdate(new \DateTime());

                $row['brand'] = $product->getBrandId();
                $row['category'] = $product->getCategoryId();

                if ($buildItems === false) {
                    $price->addPosition($this->createPosition($row, $product));
                }


                $row['status'] = $product->getName() . ' [' . $product->getId() . ']';
            }


            $items[$k] = $row;
        }

        if ($buildItems === false) {
            return $price;
        }

        return $items;
    }


    /**
     * @param array $row
     * @param $product
     * @return \Shop\Models\Position\Position
     * @throws \Propel\Runtime\Exception\PropelException
     */
    protected function createPosition(array $row, Product $product): \Shop\Models\Position\Position
    {
        $position = new \Shop\Models\Position\Position;
        $product->setdateUpdate(new \DateTime);

        $position
            ->setDesc($row['name'])
            ->setPrice($row['price'])
            ->setDestination($row['provider_city'])
            ->setProduct($product)
            ->setRemain($row['stock'])
            ->setRefId($row['id']);

        return $position;
    }

}