<?php


namespace MoySklad\Application\Modules\AdminSyncMySklad;


use MoySklad\Domain\MoySkladTask\Contracts\MoySkladEntityInterface;
use MoySklad\Models\MoySkladTask\MoySkladTask;
use MoySklad\Application\MoySkladTaskService;
use Core\Helpers\Date;

/**
 * Trait SyncMySkladTrait
 * @package MoySklad\Application\Modules\AdminSyncMySklad
 */
trait SyncMySkladTrait
{

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnTime(MoySkladTask $task): string
    {
        if (!$start = $task->getDateTimeStart()) {
            return '';
        }

        if (!$end = $task->getDateTimeEnd()) {
            $end = new \DateTime();
        }

        $time = ($end->getTimestamp() - $start->getTimestamp());

        $min = round($time / 60, 1);

        return $min > 1 ?
            ($min / 60 > 1 ? round($min / 60, 1) . ' ч' : $min . ' мин')
            : round($time) . ' сек';
    }

    /**
     * @param MoySkladTask $task
     * @return string
     */
    public function columnPriceSale(MoySkladTask $task): string
    {
        return $task->getSalePrice() ? $task->getSalePrice() : '';
    }

    /**
     * @param MoySkladTask $task
     * @return string
     */
    public function columnStock(MoySkladTask $task): string
    {
        return $task->getStockLabel();
    }

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnPrice(MoySkladTask $task): string
    {
        $scheet = $task->getPricelistSheet();

        if (!$scheet) {
            return 'авто-поиск прайс-листа по контрагенту';
        }


        return $scheet->getTitle() .
            ($scheet->getDateUpdate() ?

                ' / ' . $scheet->getDateUpdate()->format(Date::formatDisplay()) . ' (' . $scheet->getPositions()->count() . ')' : '');
    }

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnDateStart(MoySkladTask $task): string
    {
        return $task->getDateTimeStart() ?
            sprintf('<nobr>%s</nobr>', $task->getDateTimeStart()->format(Date::formatDisplay())) :
            '';
    }

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnDateEnd(MoySkladTask $task): string
    {
        return $task->getDateTimeEnd() ?
            sprintf('<nobr>%s</nobr>', $task->getDateTimeEnd()->format(Date::formatDisplay()))
            : '';
    }


    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnComplete(MoySkladTask $task): string
    {
        $taskStatus = $this->moySkladTaskService->createTaskStatus($task);

        $display = $taskStatus->getStatusDisplay();

        if ($task->getPricelistSheet() &&
            $taskStatus->getCode() === MoySkladTaskService::STATUS_FINISH_CODE &&
            $task->getDateTimeEnd() > $task->getPricelistSheet()->getDateUpdate()) {
            return '<strong title="Свежие данные, можно обновлять">' . $display . '</strong>';
        }

        return $display;
    }


    /**
     * @param MoySkladEntityInterface $task
     * @return string
     */
    public function formatActions(MoySkladEntityInterface $task): string
    {
        $items = $this->getStatusActions($task);
        //dump($this->moySkladTaskService->createTaskStatus($task));
        $items[] = '<a title="Посмотреть отчет" href="/admin/local/main/syncmysclad?report=' . $task->getId() . '">Отчет</a>';

        return $this->cBuilder->buildActionMenuAdmin($items, $task->getId())->render();
    }

    /**
     * @param MoySkladTask $task
     * @return array
     */
    protected function getStatusActions(MoySkladTask $task): array
    {
        $taskStatus = $this->moySkladTaskService->createTaskStatus($task);
        if ($taskStatus->getStatus() === 'Завис') {
            $statusAction = sprintf('<a href="/admin/local/main/syncmysclad?process=%s&force=1">Оживить загрузку</a>', $task->getId());
        } elseif ($taskStatus->getStatus() === 'Завершен') {
            $statusAction = sprintf('<a href="/admin/local/main/syncmysclad?process=%s">Просмотр данных</a>', $task->getId());
            return [
                '<nobr>' . $this->getActionLink($task) . '</nobr>',
                '<nobr>' . $statusAction . '</nobr>'
            ];
        } elseif ($taskStatus->getStatus() === 'В работе') {
            $statusAction = sprintf('<a href="/admin/local/main/syncmysclad?process=%s">Просмотр процесса загрузки</a>', $task->getId());
        } else {

            $statusAction = $this->getActionLink($task);
        }

        return ['<nobr>' . $statusAction . '</nobr>'];
    }

    protected function getActionLink(MoySkladEntityInterface $task, string $caption = 'Запустить загрузку'): string
    {
        return sprintf('<a href="/admin/local/main/syncmysclad?process=%s&first=1">' . $caption . '</a>', $task->getId());;
    }


    /**
     * @param MoySkladEntityInterface $task
     * @return string
     */
    public function columnProgress(MoySkladEntityInterface $task): string
    {
        // $report = $this->moySkladTaskService->getLastLog($task);
        $res = '';

        $file = $this->moySkladTaskService->getCachePathFinish($task->getId());
        $fileP = $this->moySkladTaskService->getCachePathProcess($task->getId());
        if (file_exists($file)) {
            $res = file_get_contents($file);
            $data = json_decode($res, true);
            $res = 'получено ' . count($data['results']) . ' из ' . $data['countAll'];

        } elseif (file_exists($fileP)) {
            $res = file_get_contents($fileP);
            $data = json_decode($res, true);

            $taskStatus = $this->moySkladTaskService->createTaskStatus($task);

            if ($taskStatus->getCode() === MoySkladTaskService::STATUS_HOVERED_CODE) {
                $img = '';
            } else {
                $img = '<img src="/assets/images/ajax-loader.gif"/>';
            }

            $res = $img . 'загружено ' . count($data['results']) . ' из ' . $data['countAll'];
        }

        return $res;
    }

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnBrand(MoySkladTask $task): string
    {
        return $task->getShopBrand()->getName();
    }

    /**
     * @param MoySkladTask $task
     * @return string
     * @throws \Propel\Runtime\Exception\PropelException
     */
    public function columnCategory(MoySkladTask $task): string
    {
        return $task->getCategory()->getNameWithParentName();
    }


    /**
     * @param MoySkladTask $task
     * @return string
     */
    public function columnInterval(MoySkladTask $task): string
    {
        if ($task->getActive() === 1) {
            return 'раз в '. str_replace(['d', 'h'], [' день', ' час'], $task->getInterval());
        }
        return '<span style=\'color: darkred\'>Нет</span>';
    }
}