<?php


namespace MoySklad\Application\Console;


use Local\Config\Main;
use MoySklad\Application\Commands\UpdatePriceListFromRemains\UpdatePriceListsByTaskCommand;
use MoySklad\Application\Commands\UpdatePriceListFromRemains\UpdatePriceListsByTaskException;
use MoySklad\Application\MoySkladFactory;
use MoySklad\Application\MoySkladService;
use MoySklad\Application\MoySkladTaskService;
use MoySklad\Infrastructure\MoySkladTask\MoySkladCriteria;
use MoySklad\Infrastructure\MoySkladTask\MoySkladTaskRepository;
use MoySklad\Models\MoySkladTask\MoySkladTask;
use spaceonfire\CommandBus\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class TaskPriceAutoLoadCommand extends Command
{
    protected $moySkladTaskRepository;
    protected $moySkladTaskService;
    protected $mainConfig;
    protected $commandBus;

    public function __construct(?string $name = null,
                                MoySkladTaskRepository $moySkladTaskRepository,
                                Main $mainConfig,
                                CommandBus $commandBus,
                                MoySkladTaskService $moySkladTaskService)
    {
        parent::__construct($name);

        $this->moySkladTaskRepository = $moySkladTaskRepository;
        $this->moySkladTaskService = $moySkladTaskService;
        $this->mainConfig = $mainConfig;
        $this->commandBus = $commandBus;
    }


    protected function configure()
    {
        $this->setName('tools:taskprices-autoload')
            ->setDescription('Auto-Loading TaskPrice MoySklad');
    }


    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Propel\Runtime\Exception\PropelException
     * @throws \Throwable
     */
    protected function execute(InputInterface $input, OutputInterface $output): void
    {

        $tasks = $this->moySkladTaskRepository->findByCriteria(
            MoySkladCriteria::create()->setFilterByActive(true)
        );

        $output->writeln(sprintf(
            'Start autoload tasks, found %s', $tasks->count()
        ));

        $run = false;

        foreach ($tasks as $task) {
            $status = $this->moySkladTaskService->createTaskStatus($task);


            switch (true) {

                case $status->getCode() === MoySkladTaskService::STATUS_FINISH_CODE &&
                    $task->getPricelistSheet() &&
                    $status->getCode() === MoySkladTaskService::STATUS_FINISH_CODE &&
                    $task->getDateTimeEnd() > $task->getPricelistSheet()->getDateUpdate():

                    //$commandBus
                    try {
                        $output->writeln(sprintf(
                            'update prices for task %s', $task->getId()
                        ));
                        $command = new UpdatePriceListsByTaskCommand($task->getId());
                        $this->commandBus->handle($command);
                        $output->writeln(sprintf(
                            'update success, count items %s', $command->getEffectedRows()
                        ));
                    } catch (UpdatePriceListsByTaskException $ex) {
                        $output->writeln($ex->getMessage());
                    }

                    sleep(10);

                    break;
                case $status->getCode() === MoySkladTaskService::STATUS_WORK_CODE:

                    $run = true;
                    break;

                case $status->getCode() === MoySkladTaskService::STATUS_HOVERED_CODE:
                    $this->moySkladTaskService->loadMoySkladTask($task);
                    $output->writeln(sprintf(
                        'task № %s is hovered!, run continuation process', $task->getId()
                    ));

                    $run = true;
                    break;
            }


        }


        $currentTime = time();

        //если не было попыток старта или нет работающих ищем задачу для автозагрузки остатков
        if ($run === false) {
            /**
             * @var MoySkladTask $task
             */
            foreach ($tasks as $task) {

                $timeEnd = $task->getDateTimeEnd() ? $task->getDateTimeEnd()->getTimestamp() : 0;


                // $currentTime - $lastTime = $hours
                //сколько прошло часов с последнего запуска
                $lastH = (int)round(($currentTime - $timeEnd) / 60 / 60);

                switch (true) {
                    case stripos($task->getInterval(), 'h'):
                        $intevalH = (int)str_replace('h', '', $task->getInterval());
                        break;
                    case stripos($task->getInterval(), 'd'):
                        $intevalH = (int)str_replace('d', '', $task->getInterval()) * 24;
                        break;

                    default:
                        $intevalH = (int)$task->getInterval();
                        break;
                }

                if ($lastH >= $intevalH) {

                    $logger = MoySkladFactory::buildLogger($this->mainConfig->getMoyskladLogin(), $currentTime, $task->getId());
                    $moySkladService = MoySkladFactory::buildService($this->mainConfig->getMoyskladLogin(), $this->mainConfig->getMoyskladPassword(), $logger);

                    $output->writeln(sprintf(
                        'task № %s is start by interval %s h', $task->getId(), $intevalH
                    ));


                    $this->moySkladTaskService->loadMoySkladTaskFirst($task, $currentTime, $moySkladService);

                    $output->writeln(sprintf(
                        'upload %s success!', $task->getId()
                    ));

                    break;
                }
            }
        }

        $output->writeln('autoload completed!');
    }

}