<?php

namespace DiamondTable;

use ActiveTable\ActionTable;
use ActiveTable\ColumnTable;
use ActiveTable\DataTableEngine;
use Core\Breadcrumb\Breadcrumb;
use Core\Concrete\RequestAwareTrait;
use Core\Contracts\RequestAwareInterface;
use Core\Form\Control\Submit;
use Core\Form\ControlBuilder;
use Diamond\Domain\Site\Contracts\SiteGetterInterface;
use Diamond\Factory\ServerRequest;
use Diamond\Repositories\Site\SiteCriteria;
use Diamond\Repositories\Site\SiteRepository;
use DiamondTable\Commands\EmptyCommand;
use DiamondTable\Commands\FormValidation;
use DiamondTable\Commands\FormView;
use DiamondTable\Commands\TableRowDelete;
use DiamondTable\Commands\TableRowExport;
use DiamondTable\Commands\TableRowSave;
use DiamondTable\Commands\TableView;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Repo\CrudRepositoryInterface;
use Repo\EntityInterface;
use Repo\PaginationInterface;

/**
 * фабрика таблиц
 * Class DataTableFactory
 * @package AutoResourceTable
 */
final class TableFactory implements LoggerAwareInterface, RequestAwareInterface
{

    use LoggerAwareTrait;
    use RequestAwareTrait;

    protected $controlBuilder;
    protected $siteRepository;
    protected $breadcrumbs;

    public function __construct(ControlBuilder $controlBuilder, SiteRepository $siteRepository, Breadcrumb $breadcrumb)
    {
        $this->logger = new NullLogger();
        $this->controlBuilder = $controlBuilder;
        $this->siteRepository = $siteRepository;
        $this->breadcrumbs = $breadcrumb;
    }


    public function buildAdminTable(CrudRepositoryInterface $repo,
                                    PaginationInterface $criteria,
                                    string $addButtonTitle = 'Добавить запись',
                                    string $name = null
    ): DataTableEngine
    {

        $name = $name ?? md5(get_class($repo));
        $request = ServerRequest::build($this->request);
        $commandManager = CommandFactory::buildFormRequest($request);
        $commandManager->setLogger($this->logger);

        //заполняет критерию нужными данными
        $criteria = SearchCriteriaBuilder::buildFromRequest($request, $criteria);

        $table =
            (new DataTableEngine($repo, $name, $commandManager, $request, $criteria))
                ->setClass('table table-striped')
                ->setAddButtonTitle($addButtonTitle)
                ->setFormClass('niffy-admin-form panel-body form-horizontal form-padding')//->setFieldsWidth(500)
        ;

        $table->setControlAccess([
            DataTableEngine::CONTROL_ACCESS_ADD,
            DataTableEngine::CONTROL_ACCESS_EDIT,
            DataTableEngine::CONTROL_ACCESS_DELETE,
            DataTableEngine::CONTROL_FILTER_BUTTON,
            DataTableEngine::CONTROL_PAGINATION,
            DataTableEngine::CONTROL_ROWS_SELECT,
            DataTableEngine::CONTROL_ROWS_ACTION,
            DataTableEngine::CONTROL_FORM_SAVE_BUTTON,
            DataTableEngine::CONTROL_FORM_CANCEL_BUTTON
        ]);

        //build commands
        $commandManager->addCommand(CommandFactory::TABLE_VIEW, new TableView($table, $addButtonTitle));
        $commandManager->addCommand(CommandFactory::FORM_VIEW, new FormView($table,$this->breadcrumbs));
        $commandManager->addCommand(CommandFactory::FORM_SAVE, new TableRowSave($table));
        $commandManager->addCommand(CommandFactory::FORM_AFTER_SAVE, new EmptyCommand($table));
        $commandManager->addCommand(CommandFactory::FORM_VALIDATE, new FormValidation($table));
        $commandManager->addCommand(CommandFactory::ROWS_DELETE, new TableRowDelete($table));
        $commandManager->addCommand(CommandFactory::ROWS_EXPORT, new TableRowExport($table));

        return $table;
    }


    /**
     * @param string $name
     * @param array $fn
     * @param string $caption
     * @return ActionTable
     */
    public function buildActionTable(string $name, array $fn, string $caption): ActionTable
    {
        return (new ActionTable(
            new Submit($name, $name), $fn
        ))->setCaption($caption);
    }


    /**
     * проверка справочника для добавления элемента управления сайтом
     * @param DataTableEngine $table
     * @throws \Repo\Concrete\Exceptions\Collection
     */
    public function prepareAdminTableForDomain(DataTableEngine $table): void
    {

        $domains = $this->siteRepository->findByCriteria(SiteCriteria::create());

        if ($domains->count() > 1) {
            $table->addField(
                $this->controlBuilder
                    ->buildDropdownUniform('domainId', $domains->getDropdownList()
                    ), true, 'Сайт', 'для многосайтовости'
            );
            $table
                ->addColumn(
                    (new ColumnTable('siteId', 'Сайт'))
                        ->setWidth(100)
                        ->setHeadAttributes([
                            'data-sort-ignore' => 'true'
                        ])
                        ->setFormat($this, 'formatDomain')
                );
        } else {
            $table->addField(
                $this->controlBuilder
                    ->buildHidden('domainId')
                    ->setValue($domains->current()->getId())
            );
        }
    }

    /**
     * @param SiteGetterInterface $template
     * @return string
     */
    public function formatDomain(SiteGetterInterface $template): string
    {
        return $template->getSite()->getName();
    }

    /**
     * @param EntityInterface $entity
     * @return string
     */
    public function formatActions(EntityInterface $entity): string
    {
        return $this->controlBuilder->buildActionMenuAdmin([], $entity->getId())->render();
    }

}