<?php

namespace DiamondTable\Commands;

use ActiveTable\DataTableEngine;
use Core\Application\Loader;
use Diamond\Application\System\SystemService;
use Diamond\Helpers\Form;
use DiamondTable\CommandException;
use DiamondTable\CommandFactory;
use Propel\Runtime\Exception\PropelException;
use Repo\Concrete\Exceptions\RepositoryException;
use Repo\EntityInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

abstract class CommandAbstract
{

    public function __construct(protected DataTableEngine $tableEngine)
    {
    }

    protected function buildEntityFromArray(array $data, ?EntityInterface $entity): EntityInterface
    {
        if (!$this->getId()) {
            $entity = $this->tableEngine->getRepo()->buildEntityFromArray($data);
        } else {
            $entity = $this->tableEngine->getRepo()->buildEntityFromArray($data, $entity);
        }

        return $entity;
    }

    public abstract function process(): void;

    protected function getPostData(): array
    {
        $data = Request::createFromGlobals()->request->all();
        array_walk($data, function (&$el) {
            $el = $el !== 'NULL' ? $el : null;
        });
        return $data;
    }

    protected function getId(): ?int
    {
        $data = Request::createFromGlobals()->request->all();
        return $data['id'] ?? null;
    }

    protected function getIds(): array
    {
        $data = Request::createFromGlobals()->request->all();
        $ids = $data['ids'] ? explode(',', $data['ids']) : [];

        return array_filter($ids,function($v){
            return !empty($v);
        });
    }

    protected function saveEntity(EntityInterface $entity)
    {
        try {
            $afterSaveEntity = clone $entity;
            $this->tableEngine->getRepo()->save($entity);
            $this->tableEngine->setTableRowEntity($afterSaveEntity);
            $this->tableEngine->getCommandFactory()->trigger(CommandFactory::FORM_AFTER_SAVE, $this->tableEngine);
        } catch (PropelException|RepositoryException $ex) {
            $res = $ex->getMessage();
            if ($ex->getPrevious()) {
                $res .= PHP_EOL . $ex->getPrevious()->getMessage();
            }
            throw new CommandException($res);
        }
    }

    protected function addFormNoticeSuccess(string $text, bool $nextCreateRowLink = false, string $messageType = 'SuccessMessage')
    {
        return $this->addFormNotice($text, $nextCreateRowLink);
    }

    protected function addFormNoticeWarning(string $text, bool $nextCreateRowLink = false, string $messageType = 'SuccessMessage')
    {
        return $this->addFormNotice($text, $nextCreateRowLink, 'ErrorMessage');
    }

    protected function addFormNotice(string $text, bool $nextCreateRowLink = false, string $messageType = 'SuccessMessage')
    {

        $method = 'build' . $messageType;

        $text = $text . ($nextCreateRowLink === true ? ' ' . anchor(adaptive_url(['fn' => 'add'], ['id']), 'Еще ' .
                    mb_strtolower($this->tableEngine->getAddButtonTitle(),'UTF-8')
                ) : '');
        Form::addTableNotice(Form::$method($text));
    }

    protected function redirect(): void
    {
        RedirectResponse::create(adaptive_url([], ['fn', 'id']))->send();
    }

    protected function redirectSelf(EntityInterface $entity): void
    {
        RedirectResponse::create(adaptive_url([
            'fn' => 'edit',
            'id' => $entity->getId()
        ]))->send();
    }
}