<?php

namespace DiamondTable\Commands;


use ActiveTable\ColumnTable;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use ActiveTable\Exceptions\TableActionException;
use ActiveTable\FormField;
use Core\Breadcrumb\Breadcrumb;
use Core\Exceptions\ForbiddenAccessException;
use Core\Form\Control\Header;
use Core\Form\Control\Hidden;
use Core\Form\Control\Button;
use Core\Form\Control\Row;
use Core\Form\Control\SubHeader;
use Core\Form\Control\Submit;
use Core\Form\Control\FormControl;
use Core\Form\Simpleform;
use DiamondTable\CommandException;
use DiamondTable\CommandFactory;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Repo\Concrete\Exceptions\ValidationException;
use Twig\Environment;
use Twig\Loader\FilesystemLoader;

/**
 * отображение формы
 * Class FormView
 * @package ActiveTable\Commands
 */
class FormView implements CommandInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    protected $engine;
    protected $breadcrumb;

    public function __construct(DataTableEngine $tableEngine, Breadcrumb $breadcrumb)
    {
        $this->engine = $tableEngine;
        $this->logger = new NullLogger();
        $this->breadcrumb = $breadcrumb;
    }

    /**
     * @return void
     * @throws TableActionException
     */
    public function process(): void
    {
        $form = new Simpleform($this->engine->getName());
        $form->disableNativeValidation();
        $form->setTwig(new Environment(new FilesystemLoader()));
        $form->setAction(adaptive_url([]));
        $form->setClass($this->engine->getFormClass());

        //ищем в справочнике нужное действие
        $form->addValidationSuccessEvent([$this, 'validationSuccess']);
        $form->addValidationPrepareEvent([$this, 'validationProcess']);

        $rowData = $requires = [];

        $queryParams = $this->engine->getRequest()->getQueryParams();

        $id = $this->engine->getCriteria()->getfilterById();

        //редактирование
        if (isset($queryParams['fn']) && $queryParams['fn'] === 'edit' && $id) {
            $entity = $this->engine->loadFormEntity($id);

            if (!$entity) {
                throw new ForbiddenAccessException('row not found by id ' . $id);
            }

            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $headerText = $this->engine->getFormEditText() . $id;
            $form
                ->addField(
                    new Hidden('id', input_get('id'))
                );
        } //копирование
        elseif (isset($queryParams['fn']) && $queryParams['fn'] === 'add' && $id) {
            $entity = $this->engine->loadFormEntity($id);
            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $headerText = $this->engine->getFormAddText();
        } //новая
        else {
            $entity = null;
            $headerText = $this->engine->getFormAddText();
        }


        $this->breadcrumb
            ->addCrumb($headerText);

        $foundHeader = false;

        foreach ($this->engine->getFields() as $field) {
            if (is_a($field->getControl(), Header::class)) {
                $foundHeader = true;
                break;
            }
        }

        if ($foundHeader === false) {
            $form->addHeader($headerText);
        }

        $postData = $this->engine->getRequest()->getParsedBody();

        /**
         * @var FormField $formControl
         * @var FormControl $control
         * @var ColumnTable $column
         */
        foreach ($this->engine->getFields() as $formControl) {

            $control = $formControl->getControl();
            $name = strtolower($control->getName());

            $caption = $formControl->getCaption();

            if (is_a($control, \Core\Form\Control\FormField::class)) {

                $control->addClass('form-control');
                if (isset($postData[$name])) {
                    $control->setValue($postData[$name]);
                } elseif ($entity && (is_callable($control->getFn()))) {
                    $control->setValue(call_user_func($control->getFn(), $entity));
                } elseif (isset($rowData[$name])) {
                    $control->setValue($rowData[$name]);
                }

                if ($formControl->isRequire() === true) {
                    $requires[] = $control->getName();
                }

                //добавшляем контрол в форму
                $form->addField(
                    $control,
                    $caption,
                    $formControl->isRequire() === true ? 'required' : '',
                    $formControl->getHelpCaption()
                );
            } else {
                $form->addControl($control);
            }
        }


        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_SAVE_BUTTON)) {
            $form
                ->addControl(
                    (new Submit('submit', 'Сохранить'))
                        ->setClass('btn btn-success')//btn-info
                );
/*            $form
                ->addControl(
                    (new Submit('submit', 'Сохранить и закрыть'))
                        ->setClass('btn btn-success')
                );*/
        }

        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_CANCEL_BUTTON)) {

            $form->addControl(
                (new Button('cancel', 'Отмена'))
                    ->setType('button')
                    ->setClass('btn btn-default')
                    ->setOnClick("location.href='" . adaptive_url([], ['fn', 'id']) . "'")
            );
        }

        if(!$template = $this->engine->getFormTemplate()){
            $template = $this->getTemplateForm($form, $requires, $this->engine->getTabs(), $this->engine->getFieldGroups());
        }

        $form->setCustomTpl($template);

        $this->engine->addContent($form->render());
    }


    private function createHtmlRowForm(FormControl $field, array $requires, array $groups): string
    {
        $fName = $this->prepareFieldName($field->getName());
        $html = '';
        if (is_a($field, Row::class)) {

            $html .= sprintf('<div class="stdformrow">{{%s.control}}</div>', $field->getName());

        } elseif (is_a($field, SubHeader::class)) {

            $html .= sprintf('<div class="bord-btm pad-ver text-main text-bold">{{%s.control}}</div>', $fName);

        } elseif (is_a($field, Button::class) || is_a($field, Hidden::class) || is_a($field, Header::class)) {
        } else {

            if (in_array($field->getName(), $requires)) {
                $required_class = 'required';
            } else {
                $required_class = '';
            }
            $group = '';
            foreach ($groups as $name => $groupItems) {
                if (in_array($field->getName(), $groupItems)) {
                    $group = $name;
                    break;
                }
            }

            $html .= sprintf(
                '<div class="form-group %s %s">{{%s.label}}<div class="col-md-9">{{%s.control}}</div></div>',
                $group, $required_class, $fName, $fName
            );
        }
        return $html;
    }


    private function getTemplateForm(Simpleform $form, array $requires, array $tabs, array $groups): string
    {

        $html = '';

        if (!empty($form->getFormHtmlTop())) {
            $html .= $form->getFormHtmlTop();
        }


        if (count($tabs)) {
            $i = 0;
            $names = [];
            $html .= '<div class="tab-content">';
            foreach ($tabs as $tab => $tabFields) {
                $html .= '<div class="tab-pane fade in ' . ($i === 0 ? 'active' : '') . '" id="tab-' . md5($tab) . '">';
                foreach ($tabFields as $tabField) {
                    $field = $form->findFieldByName($tabField);
                    $html .= $this->createHtmlRowForm($field, $requires, $groups);
                    $names[$tabField] = true;
                }
                $html .= '</div>';
                $i++;
            }
            $html .= '</div>';

            foreach ($form->getFields() as $field) {
                if (isset($names[$field->getName()])) continue;
                $html .= $this->createHtmlRowForm($field, $requires, $groups);
            }

        } else {
            foreach ($form->getFields() as $field) {
                $html .= $this->createHtmlRowForm($field, $requires, $groups);
            }
        }


        $tpl_html_header = $tpl_html_footer = '';


        foreach ($form->getFields() as $k => $field) {

            $fName = $this->prepareFieldName($field->getName(), $k);

            if (is_a($field, Header::class) && !is_a($field, SubHeader::class)) {
                $panelControl = '';

                if (count($tabs) > 0) {
                    $panelControl = '<div class="panel-control"><ul class="nav nav-tabs">';
                    $k = 0;
                    foreach ($tabs as $tab => $fields) {
                        $panelControl .= '<li ' . ($k === 0 ? 'class="active"' : '') . '><a href="#tab-' . md5($tab) . '" data-toggle="tab">' . $tab . '</a></li>';
                        $k++;
                    }
                    $panelControl .= '</ul></div>';
                }

                $tpl_html_header = sprintf('<div class="panel-heading">%s<h3 class="panel-title">%s</h3></div>', $panelControl, $field->getValue());

            } elseif (is_a($field, Button::class) || is_a($field, Hidden::class)) {

                $tpl_html_footer .= sprintf('{{%s.control}}', $fName);
            }
        }

        if (!empty($tpl_html_footer)) {
            $tpl_html_footer = '<div class="panel-footer text-right">' . $tpl_html_footer . '</div>';
        }

        if (!empty($form->getFormHtmlBootom())) {
            $html .= $form->getFormHtmlBootom();
        }

        $html = '<div class="panel">' . $tpl_html_header . '<div class="panel-body">' . $html . '</div>' . $tpl_html_footer . '</div>';
        return $html;
    }

    private function prepareFieldName(string $fName): string
    {
        if (strpos($fName, '[') !== false) {
            return str_replace(['[', ']'], '', $fName) . '_';
        }
        return $fName;
    }


    /**
     * @param array $data
     * @param Simpleform $form
     * @return array|null
     */
    public function validationProcess(array $data, Simpleform $form): ?array
    {

        try {
            $this->engine->getCommandFactory()->trigger(CommandFactory::FORM_VALIDATE, $this->engine);
        } catch (ValidationException $ex) {
            foreach ($ex->getErrors() as $error) {
                $form->addCustomMessage($error, Simpleform::MESSAGE_ERROR);
            }
            return null;
        }

        return $data;
    }

    /**
     * @param array $data
     * @param Simpleform $form
     */
    public function validationSuccess(array $data, Simpleform $form): void
    {
        try {
            $this->engine->getCommandFactory()->trigger(CommandFactory::FORM_SAVE, $this->engine);
        } catch (CommandException $ex) {
            if (getenv('MODE') === 'dev') {
                throw $ex;
            }
            $this->logger->error($ex->getMessage());
            $newForm = clone $form;
            $form->addCustomMessage('При сохранении возникла ошибка', Simpleform::MESSAGE_ERROR);
            $form->addBuffer($newForm->generateForm());
        }
    }
}