<?php

namespace DiamondTable;

use ActiveTable\Contracts\CommandFactoryInterface;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use ActiveTable\Exceptions\ActiveTableException;
use DiamondTable\Commands\FormView;
use DiamondTable\Commands\TableAction;
use DiamondTable\Commands\TableView;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Фабрика комманд
 * Class CommandFactory
 * @package ActiveTable\Factories
 */
final class CommandFactory implements CommandFactoryInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    public const TABLE_VIEW = 'table_view';
    public const FORM_VIEW = 'form_view';
    public const FORM_SAVE = 'form_save';
    public const FORM_VALIDATE = 'form_validate';
    public const ROWS_EXPORT = 'rows_export';
    public const ROWS_DELETE = 'rows_delete';
    //public const ROW_DELETE = 'row_delete';

    /**
     * Тригер - событие для запуска комманды
     * @var array
     */
    protected $triggers = [
        self::TABLE_VIEW,
        self::FORM_VIEW,
        self::FORM_SAVE,
        self::ROWS_DELETE,
        self::ROWS_EXPORT,
        self::FORM_VALIDATE
    ];

    protected $commands = [];

    /**
     * @var string
     */
    protected $trigger;

    /**
     * @return string
     */
    public function getEventName(): string
    {
        return $this->trigger;
    }

    /**
     * @param string $trigger
     * @param CommandInterface $command
     * @return $this|CommandFactoryInterface
     */
    public function addCommand(string $trigger, CommandInterface $command)
    {
        $this->commands[$trigger] = $command;
        $this->triggers []= $trigger;
        $this->triggers = array_unique($this->triggers);
        return $this;
    }


    /**
     * @param ServerRequestInterface $request
     * @return CommandFactory
     */
    public static function buildFormRequest(ServerRequestInterface $request): self
    {

        $postData = $request->getParsedBody();
        $getData = $request->getQueryParams();

        if (
            isset($postData['actiontype']) && !empty($postData['actiontype'])
        ) {

            return new self($postData['actiontype']);
        } elseif (
            isset($postData['submit']) ||
            (isset($getData['fn']) && in_array($getData['fn'], ['add', 'edit']))
        ) {
            return new self(self::FORM_VIEW);
        }


        return new self();
    }

    /**
     * CommandFactory constructor.
     * @param string $trigger
     */
    public function __construct(string $trigger = self::TABLE_VIEW)
    {

        $this->trigger = $trigger;
        $this->logger = new NullLogger();
    }

    /**
     * Возвращаем комманды в зависимости от тригера-комманды
     * @param DataTableEngine $tableEngine
     * @return CommandInterface
     * @throws ActiveTableException
     */
    public function build(DataTableEngine $tableEngine): CommandInterface
    {

        if (!in_array($this->trigger, $this->triggers, true)) {
            throw new ActiveTableException(sprintf('trigger `%s` not found', $this->trigger));
        }

        foreach ($this->commands as $triger => $command) {
            if ($this->trigger === $triger) {
                if(is_a($command,LoggerAwareInterface::class)){
                    $command->setLogger($this->logger);
                }

                return $command;
            }
        }

        throw new \RuntimeException(sprintf('Commands `%s` not found',$this->trigger));
    }

    /**
     * @param string $name
     * @param DataTableEngine $engine
     * @throws ActiveTableException
     */
    public function trigger(string $name, DataTableEngine $engine): void
    {
        $this->trigger = $name;
        self::build($engine)->process();
    }

}