<?php
/**
 * Created by PhpStorm.
 * User: Admin
 * Date: 19.08.2020
 * Time: 22:39
 */

namespace DiamondTable\Commands;


use ActiveTable\ColumnTable;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use Core\Form\Control\Button;
use Core\Form\Control\Dropdown;
use Core\Form\Control\Hidden;
use Core\Form\Control\Input;
use Core\Helpers\Text;
use Diamond\Helpers\Url;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Repo\EntityInterface;
use Symfony\Component\HttpFoundation\Session\Session;

class TableView implements CommandInterface, LoggerAwareInterface
{
    use LoggerAwareTrait;

    protected $engine;
    protected $summary;
    protected $defaultRows;
    protected $listRows = [25, 50, 100, 200, 500];

    public function __construct(DataTableEngine $tableEngine)
    {
        $this->engine = $tableEngine;
        $this->logger = new NullLogger();
        $this->defaultRows = $this->listRows[0];
    }

    /**
     * main process
     */
    public function process(): void
    {
        $this->engine->addContent($this->buildMessages());
        $this->engine->addContent($this->buildTopControls());
        $this->engine->addContent($this->buildTable());
        $this->engine->addContent($this->buildPagination());
    }

    /**
     * @return string
     */
    private function buildMessages(): string
    {
        $session = new Session();
        $html = '';
        foreach ($session->getFlashBag()->get('table-notice') as $notice) {
            $html .= $notice;
        }
        return $html;
    }

    /**
     * @return string
     */
    private function buildTopControls(): string
    {
        $html = '';

        foreach ($this->engine->getButtons() as $button) {
            if ($this->engine->hasControlAccess($button->getName())) {
                $html .= $button->render();
//                anchor(adaptive_url(['fn'=>'edit']),'<span>Добавить</span>',[
//                    'class'=>'btn btn_document'
//                ]);
            }
        }

        $globalControls = '';

        $queryParams = $this->engine->getRequest()->getQueryParams();
        $session = new Session();
        $rowsValue = $queryParams['rows'] ?? null;
        $rowsValue = $session->get('table-rows') ?? $rowsValue;


        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_ROWS_SELECT)) {
            $globalControls .=
                (new Dropdown('rows', array_combine(array_values($this->listRows), $this->listRows)))
                    ->setClass('radius3')
                    ->setValue($rowsValue)
                    ->onChange(sprintf(
                        "let url='%s';location.href=url.replace('ROWS',this.value)",
                        adaptive_url(['rows' => 'ROWS'])
                    ))
                    ->render();
        }

        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_FILTER_BUTTON)
            && count($this->engine->getFilters())
        ) {
            $globalControls .=
                (new Button('filter_button',
                    isset($queryParams['filter']) ? 'Скрыть фильтр' : 'Показать фильтр'
                ))
                    ->setType('button')
                    ->setOnClick('filter_show(this)')
                    ->render();
        }

        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_ROWS_ACTION) && count($this->engine->getRowActions())) {

            $globalControls .= '<select name="actiontype" class="radius3"><option value="">-выберите действие-</option>';

            foreach ($this->engine->getRowActions() as $action) {
                $globalControls .= sprintf(
                    '<option value="%s">%s</option>',
                    $action->getControl()->getName(),
                    $action->getCaption()
                );
            }

            $globalControls .= '</select>' . $this->renderActionButtons();
        } elseif ($globalControls) {
            $globalControls .= '<input name="actiontype" type="hidden">' . $this->renderActionButtons();
        }


        if ($globalControls || $html) {
            $htmlResult = '<div class="tableoptions table-control">
        <div class="dataTables_wrapper">
            <div class="two_third">
            <form class="diamond-table-action" onsubmit="return false" method="post" name="action-selected-rows">
                    ' . $globalControls . '
            <input type="hidden" name="ids" /></form>
            </div>

            <div class="one_third last right">
                ' . $html . '
            </div>
        </div>
    </div>';
        } else {
            $htmlResult = '';
        }

        return $htmlResult;
    }

    /**
     * @return string
     */
    private function renderActionButtons(): string
    {
        $s = '';
        foreach($this->engine->getActionButtons() as $button){

            if($button->getName() === 'action_global' && $this->engine->hasControlAccess(DataTableEngine::CONTROL_ROWS_ACTION) === false ){
                continue;
            }

            $s .= $button->render();
        }

        //dd($this->engine->getActionButtons());
        //$s = '<button name="action_global" type="submit" class="radius3">Применить</button>';
        return $s;
    }

    /**
     * @return string
     */
    private function buildPagination(): string
    {
        $info = $nav = $controls = '';

        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_PAGINATION)) {
            $criteria = $this->engine->getCriteria();

            $page = $this->engine->getCriteria()->getPage();

            $limit = $this->engine->getCriteria()->getLimit();

            $count = $this->engine->getRepo()->count($criteria);

            while (true) {

                if ($page > 1 && $count < $page * $limit && $count < $limit) {
                    $page--;
                    continue;
                }

                break;
            }

            // кол-во на странице
            $limit = $limit > $count ? $count : $limit;

            //кол-во страниц
            $allPages = $count > $limit ? (int)ceil($count / $limit) : 1;

            //текущая страница
            $page = $page < 0 ? 1 : $page;

            //показано записей
            $offset = $page * $limit;
            $offset = $offset > $count ? $count : $offset;

            $from = ($offset - $limit + 1);
            $to = ($offset > $count ? $count : $offset);

            // dump($from,$to,$count,$offset);

            if ($from === 1 && $to === $count) {
                $view = $to . ' запис' . Text::morph($count, 'ь', 'и', 'ей');
            } else {
                $view = $from . '-' . $to . ' из ' . $count . ' запис' . Text::morph($count, 'ь', 'ей', 'ей');
            }


            $info = 'Показан' . Text::morph($to, 'а', 'о', 'о') . ' ' . $view;

            if ($page > 1) {
                $bback = Url::anchor(adaptive_url(['page' => 1]), '<i><<</i>',
                    ['class' => 'first paginate_button']);
                $back = Url::anchor(adaptive_url(['page' => $page - 1]), '<i><</i>',
                    ['class' => 'previous paginate_button']);
            } else {
                $bback = $back = '';
            }


            if ($page !== $allPages) {
                $nnext = Url::anchor(adaptive_url(['page' => $allPages]), '<i>>></i>', ['class' => 'next paginate_button']);
                $next = Url::anchor(adaptive_url(['page' => $page + 1]), '<i>></i>', ['class' => 'last paginate_button']);
            } else {
                $nnext = $next = '';
            }

            $navigation = '<span class="navigation"><form action="' . adaptive_url([], ['page']) .
                '" name="pager">Страница ' . (new Input('page'))
                    ->setValue($page)
                    ->setClass('radius3 pager')
                    ->setWidth(20)
                    ->render()
                . ' из ' . $allPages;

            $queryData = $this->engine->getRequest()->getQueryParams();

            foreach ($queryData as $queryName => $value){
                if($queryName !== 'page'){
                    $navigation .= (new Hidden($queryName, $value))->render();
                }
            }
            

            $navigation .= '</form></span>';

            if ($allPages > 1) {
                $nav = $bback . ' ' . $back . $navigation . ' ' . $next . ' ' . $nnext;;
            } else {
                $nav = nbs();
            }


        }

        if ($info || $nav) {
            $html = '<div class="dataTables_wrapper">
        <div class="dataTables_info" id="dyntable_info">' . $info . '</div>
        <div class="dataTables_paginate paging_full_numbers" id="dyntable_paginate">' . $nav . '</div></div>';
        } else {
            $html = '';
        }

        return $html;
    }

    /**
     * @return string
     */
    private function buildTable(): string
    {

        $criteria = $this->engine->getCriteria();

        //set default sort
        $useSort = false;
        foreach (get_class_methods($criteria) as $method) {
            if (strpos($method, 'getSortBy') !== false && $criteria->{$method}() !== null) {
                $useSort = true;
            }
        }

        if ($useSort === false && $this->engine->getDefaultSortColumn()) {
            $method = 'setSortBy' . array_key_first($this->engine->getDefaultSortColumn());
            $criteria->{$method}(array_values($this->engine->getDefaultSortColumn())[0]);
        }

        $rows = $this->engine->getRepo()->findByCriteria($criteria);
        $getData = $this->engine->getRequest()->getQueryParams();

        $html = '<form name="main-table" method="get" action="' . adaptive_url() . '">' .
            '<table class="' . $this->engine->getClass() . '" cellspacing="0" cellpadding="0" border="0">';

        $columns = $this->engine->getColumns();

        if ($this->engine->hasControlAccess(DataTableEngine::CONTROL_ROWS_ACTION) === false) {
            foreach ($columns as $k => $column) {
                if ($column->getName() === 'check') {
                    unset($columns[$k]);
                    break;
                }
            }
        }
        //расчет процента для неотмеченных колонок
        $percent = 100;
        $col = 0;
        foreach ($columns as $column) {
            $w = $column->getWidth();
            if (strpos($w, '%') !== false) {
                $percent -= rtrim($w, '%');
            } else {
                $col++;
            }
        }

        //анализ колонок на остутсвующие указания ширины, если найедены то игнорируем автоподгон
        $fixCount = false;
        foreach ($columns as $column) {
            if ((int)$column->getWidth() === 0) {
                $fixCount = true;
                break;
            }
        }

        if ($percent !== 0 && $fixCount === false) {

            $newWidth = round(abs($percent) / count($columns));

            foreach (array_reverse($columns) as $column) {

                if ($percent === 0) {
                    break;
                }

                $width = (int)rtrim($column->getWidth(), '%');

                if ($percent < 0) {
                    $column->setWidth($width - $newWidth, true);
                    $percent += $newWidth;
                } else {
                    $column->setWidth($width + $newWidth, true);
                    $percent -= $newWidth;
                }

            }
        }


        $htmlCollumns = '<colgroup>';
        $htmlHeader = '<thead><tr>';
        $htmlFotter = '<tfoot><tr>';
        $i = 0;


        /**
         * @var ColumnTable $column
         */
        foreach ($columns as $key => $column) {
            $i++;

            $caption = $column->getCaption() ?? $column->getName();
            $num = ($i % 2 === 0 ? 1 : 0);
            $ignoreSort = array_filter(array_keys($getData), function ($el) {
                return strpos($el, 'sort') !== false;
            });

            if ($column->isSorted() === true) {

                $sortParam = 'sort_by_' . $column->getName();
                $sortRule = 'asc';
                $sortClass = '/images/sort_both_white.png';

                if (isset($getData[$sortParam]) && $getData[$sortParam] === 'asc') {
                    $sortRule = 'desc';
                    $sortClass = '/images/sort_asc_white.png';
                } elseif (isset($getData[$sortParam]) && $getData[$sortParam] === 'desc') {
                    $sortRule = 'asc';
                    $sortClass = '/images/sort_desc_white.png';
                }

                $findKey = array_search($sortParam, $ignoreSort, true);

                if ($findKey !== false) {
                    unset($ignoreSort[$findKey]);
                }
                $ignoreSort [] = 'page';

                $url = adaptive_url([
                    $sortParam => $sortRule
                ], $ignoreSort);

                $caption = '<nobr>' . $caption . '<a href="' . $url . '"><img src="/_diamond/diamond-cms/assets/' . $sortClass . '" /></a></nobr>';
            }

            $htmlHeader .= '<td class="head' . $num . '">' . $caption . '</td>';
            $htmlFotter .= '<td class="head' . $num . '">' . $caption . '</td>';

            if (stripos($column->getWidth(), 'px') !== false) {
                $style = sprintf('style="width:%s"', $column->getWidth());
            } elseif (stripos($column->getWidth(), '%') !== false) {

                $calculatePercent = ($fixCount === false && $col > 0 ? (round($percent / $col) . '%') : $column->getWidth());
                $style = sprintf('style="width:%s"', $calculatePercent);
            } else {
                $style = '';
            }


            $htmlCollumns .= '<col ' . $style . ' class="con' . $num . '">';
        }
        $htmlCollumns .= '</colgroup>';
        $htmlHeader .= '</tr></thead>';
        $htmlFotter .= '</tr></tfoot>';

        $html .= $htmlCollumns;
        $html .= $htmlHeader;
        $html .= $htmlFotter;
        $html .= '<tbody>';

        //filters

        $filters = $this->engine->getFilters();

        if (count($filters)) {

            $style = isset($getData['filter']) ? '' : 'style="display: none"';

            $html .= '<tr ' . $style . ' class="table-filter">';

            $forIgnore = ['filter'];

            foreach ($filters as $filter) {
                $filterName = $filter->getControl()->getName();
                $forIgnore[] = $filterName;
            }


            foreach ($columns as $key => $column) {
                $html .= '<td>';
                foreach ($filters as $filter) {

                    $filterName = $filter->getControl()->getName();

                    if (in_array($filterName, ['search_by_' . $column->getName(), 'filter_by_' . $column->getName()], true)) {

                        $html .=
                            $filter
                                ->getControl()
                                ->setValue($getData[$filterName] ?? null)
                                ->render();
                        break;
                    } elseif ($column->getName() === 'actions') {
                        $html .=
                            '<nobr><button value="1" type="submit" name="filter" class="search-button stdbtn btn_red">Поиск</button>' .
                            nbs() .
                            '<button onclick="location.href=\'' . adaptive_url([], $forIgnore) . '\'" type="button" class="search-button stdbtn btn_black">Сброс</button>
                            </nobr>';
                        break;
                    }
                }
                $html .= '</td>';
            }
            $html .= '</tr>';
        }


        $i = 0;

        /**
         * @var EntityInterface $entity
         */
        foreach ($rows as $entity) {
            $i++;
            $id = $entity->getId();
            $html .= '<tr ' . ($id > 0 ? 'id="row' . $id . '"' : '') . '>';
            foreach ($columns as $key => $column) {
                $html .= '<td>' . $this->prepareRow($entity, $column) . '</td>';
            }
            $html .= '</tr>';
        }

        $html .= '</tbody>';
        $html .= '</table></form>';
        return $html;
    }

    /**
     * @param $model
     * @param ColumnTable $column
     * @return mixed|string
     */
    protected function prepareRow($model, ColumnTable $column)
    {

        $resultReal = method_exists($model, 'get' . $column->getName()) ?
            $model->{'get' . $column->getName()}() : '&nbsp;';

        $callable = $column->getFormat();

        if (is_callable($callable) || (is_array($callable) && isset($callable[0]) && is_object($callable[0]))) {
            $result = call_user_func($callable, $model);
        } else {
            $result = $resultReal;
        }


//        if ($resultReal > 0 && isset($this->summary[$column->getName()])) {
//            $this->summary[$column->getName()] += (float)$resultReal;
//        }

        return $result;
    }
}