<?php

namespace DiamondTable\Commands;


use ActiveTable\ColumnTable;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use ActiveTable\Exceptions\TableActionException;
use ActiveTable\FormField;
use Core\Form\Control\Hidden;
use Core\Form\Control\Button;
use Core\Form\Control\Submit;
use Core\Form\Control\FormControl;
use Core\Form\Simpleform;
use DiamondTable\CommandException;
use DiamondTable\CommandFactory;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Repo\Concrete\Exceptions\ValidationException;
use Twig\Environment;
use Twig\Loader\FilesystemLoader;

/**
 * отображение формы
 * Class FormView
 * @package ActiveTable\Commands
 */
class FormView implements CommandInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    protected $engine;

    public function __construct(DataTableEngine $tableEngine)
    {
        $this->engine = $tableEngine;
        $this->logger = new NullLogger();
    }

    /**
     *
     */
    public function process(): void
    {
        $form = new Simpleform($this->engine->getName());
        $form->setTwig(new Environment(new FilesystemLoader()));
        $form->setAction(adaptive_url([]));
        $form->setClass($this->engine->getFormClass());

        //ищем в справочнике нужное действие
        $form->addValidationSuccessEvent([$this, 'validationSuccess']);
        $form->addValidationPrepareEvent([$this, 'validationProcess']);

        $rowData = [];

        $queryParams = $this->engine->getRequest()->getQueryParams();

        $id = $this->engine->getCriteria()->getfilterById();

        //редактирование
        if(isset($queryParams['fn']) && $queryParams['fn']==='edit' && $id){
            $entity = $this->engine->getRepo()->findById($id);
            
            if(!$entity){
                throw new TableActionException('row not found by id ' . $id);
            }
            
            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $headerText = 'Редактирование записи №' . $id;
            $form
                ->addField(
                    new  Hidden('id')
                );
        }
        //копирование
        elseif(isset($queryParams['fn']) && $queryParams['fn']==='add' && $id){
            $entity = $this->engine->getRepo()->findById($id);
            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $headerText ='Новая запись';
        }
        //новая
        else{
            $entity = null;
            $headerText = 'Новая запись';
        }

        $foundHeader = false;

        foreach ($this->engine->getFields() as $field){
            if($field->getControl()->getName()==='header'){
                $foundHeader = true;
                break;
            }
        }

        if($foundHeader === false){
            $form->addHeader($headerText);
        }

        $postData = $this->engine->getRequest()->getParsedBody();


        /**
         * @var FormField $formControl
         * @var FormControl $control
         * @var ColumnTable $column
         */
        foreach ($this->engine->getFields() as $formControl) {

            $control = $formControl->getControl();
            $name = strtolower($control->getName());

            $caption = $formControl->getCaption();

            if (is_a($control, \Core\Form\Control\FormField::class)) {
                
                if(isset($postData[$name])){
                    $control->setValue($postData[$name]);
                }
                elseif($entity && (is_callable($control->getFn()))){
                    $control->setValue(call_user_func($control->getFn(), $entity));
                }
                elseif (isset($rowData[$name])) {
                    $control->setValue($rowData[$name]);
                }

                $form->addField(
                    $control,
                    $caption,
                    $formControl->isRequire() === true ? 'required' : '',
                    $formControl->getHelpCaption()
                );
            } else {
                $form->addControl($control);
            }
        }


        if($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_SAVE_BUTTON)){
            $form
                ->addControl(
                (new Submit('submit', 'Сохранить'))
                    ->setClass('stdbtn btn_orange')
            );
        }

        if($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_CANCEL_BUTTON)){

            $form->addControl(
                (new Button('cancel', 'Отмена'))
                    ->setType('button')
                    ->setClass('stdbtn')
                    ->setOnClick("location.href='" . adaptive_url([], ['fn', 'id']) . "'")
            );
        }


        $this->engine->addContent($form->render() );
    }


    /**
     * @param array $data
     * @param Simpleform $form
     * @return array|null
     */
    public function validationProcess(array $data, Simpleform $form): ?array
    {

        try{
            $this->engine->getCommandFactory()->trigger(CommandFactory::FORM_VALIDATE,$this->engine);
        }
        catch (ValidationException $ex){

            foreach ($ex->getErrors() as $error){
                $form->addCustomMessage($error,Simpleform::MESSAGE_ERROR);
            }
            return null;
        }

        return $data;
    }

    /**
     * @param array $data
     * @param Simpleform $form
     */
    public function validationSuccess(array $data, Simpleform $form):void
    {
        try{
            $this->engine->getCommandFactory()->trigger(CommandFactory::FORM_SAVE,$this->engine);
        }
        catch (CommandException $ex){
            $this->logger->error($ex->getMessage());
            $newForm = clone $form;
            $form->addCustomMessage('При сохранении возникла ошибка', Simpleform::MESSAGE_ERROR);
            $form->addBuffer($newForm->generateForm());
        }
    }
}