<?php

namespace DiamondTable;

use ActiveTable\ActionTable;
use ActiveTable\ColumnTable;
use ActiveTable\DataTableEngine;
use Core\Concrete\RequestAwareTrait;
use Core\Contracts\RequestAwareInterface;
use Core\Form\Control\Button;
use Core\Form\Control\Submit;
use Core\Form\ControlBuilder;
use Diamond\Domain\Site\Contracts\SiteGetterInterface;
use Diamond\Factory\ServerRequest;
use Diamond\Repositories\Site\SiteCriteria;
use Diamond\Repositories\Site\SiteRepository;
use DiamondTable\Commands\TableExport;
use DiamondTable\Commands\TableRowDelete;
use DiamondTable\Commands\TableRowSave;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Repo\CrudRepositoryInterface;
use Repo\EntityInterface;
use Repo\PaginationInterface;

/**
 * фабрика таблиц
 * Class DataTableFactory
 * @package AutoResourceTable
 */
final class TableFactory implements LoggerAwareInterface, RequestAwareInterface
{

    use LoggerAwareTrait;
    use RequestAwareTrait;

    protected $controlBuilder;
    protected $siteRepository;

    public function __construct(ControlBuilder $controlBuilder, SiteRepository $siteRepository)
    {
        $this->logger = new NullLogger();
        $this->controlBuilder = $controlBuilder;
        $this->siteRepository = $siteRepository;
    }

    /**
     * @param CrudRepositoryInterface $repo
     * @param PaginationInterface $criteria
     * @param string $addButtonTitle
     * @param string|null $name
     * @return DataTableEngine
     * @throws \ActiveTable\Exceptions\ActiveTableException
     */
    public function buildAdminTable(CrudRepositoryInterface $repo,
                                    PaginationInterface $criteria,
                                    string $addButtonTitle = 'Добавить запись',
                                    string $name = null
    ): DataTableEngine
    {

        $name = $name ?? md5(get_class($repo));
        $request = ServerRequest::build($this->request);
        $commandFactory = CommandFactory::buildFormRequest($request);
        $commandFactory->setLogger($this->logger);

        //заполняет критерию нужными данными
        $criteria = SearchCriteriaBuilder::buildFromRequest($request, $criteria);

        $table =
            (new DataTableEngine($repo, $name, $commandFactory, $request, $criteria))
                ->setClass('stdtable stdtablecb')
                ->setFormClass('stdform stdform2')//->setFieldsWidth(500)
        ;
        $table->setControlAccess([
            DataTableEngine::CONTROL_ACCESS_ADD,
            DataTableEngine::CONTROL_ACCESS_EDIT,
            DataTableEngine::CONTROL_ACCESS_DELETE,
            DataTableEngine::CONTROL_FILTER_BUTTON,
            DataTableEngine::CONTROL_PAGINATION,
            DataTableEngine::CONTROL_ROWS_SELECT,
            DataTableEngine::CONTROL_ROWS_ACTION,
            DataTableEngine::CONTROL_FORM_SAVE_BUTTON,
            DataTableEngine::CONTROL_FORM_CANCEL_BUTTON
        ]);

        $controlFactory = new ControlFactory();

        if ($table->hasControlAccess(DataTableEngine::CONTROL_ROWS_SELECT)) {
            $table->addColumn(
                (new ColumnTable('check', '<input class="checkall" type="checkbox">'))
                    ->setWidth(4, true)
                    ->setFormat($controlFactory, 'buildCheckbox')
            );
        }

//        $table->addRowActionTable(
//            self::buildActionTable('delete', [new TableRowDelete($table), 'process'])
//        );
//
//        $table->addRowActionTable(
//            self::buildActionTable('add', [new TableRowSave($table), 'process'])
//        );
//
//        $table->addRowActionTable(
//            self::buildActionTable('edit', [new TableRowSave($table), 'process'])
//        );
//
//        $table->addRowActionTable(
//            self::buildActionTable('exportTable', [new TableExport($table), 'process'])
//        );


        $table
            ->addButton(
                (new Button('add', $addButtonTitle))
                    ->setOnClick("location.href='" . adaptive_url(['fn' => 'add']) . "'")
            );

        $table
            ->addRowActionTable(self::buildActionTable('delete-selected', [$this, 'deleteSelected'], 'Удалить'));

        return $table;
    }

    /**
     * @param string $name
     * @param array $fn
     * @param string $caption
     * @return ActionTable
     */
    public static function buildActionTable(string $name, array $fn, string $caption): ActionTable
    {
        return (new ActionTable(
            new Submit($name, $name), $fn
        ))->setCaption($caption);
    }

    /**
     * @param DataTableEngine $engine
     */
    public function deleteSelected(DataTableEngine $engine)
    {

        $action = new TableRowDelete($engine);
        $action->setLogger($this->logger);
        $action->process();
    }

    /**
     * проверка справочника для добавления элемента управления сайтом
     * @param DataTableEngine $table
     * @throws \Repo\Concrete\Exceptions\Collection
     */
    public function prepareAdminTableForDomain(DataTableEngine $table): void
    {

        $domains = $this->siteRepository->findByCriteria(SiteCriteria::create());

        if ($domains->count() > 1) {
            $table->addField(
                $this->controlBuilder
                    ->buildDropdownUniform('domainId', $domains->getDropdownList()
                    ), true, 'Сайт', 'для многосайтовости'
            );
            $table
                ->addColumn(
                    (new ColumnTable('domain', 'Сайт'))
                        ->setWidth(15, true)
                        ->setFormat($this, 'formatDomain')
                );
        } else {
            $table->addField(
                $this->controlBuilder
                    ->buildHidden('domainId')
                    ->setValue($domains->current()->getId())
            );
        }
    }

    /**
     * @param SiteGetterInterface $template
     * @return string
     */
    public function formatDomain(SiteGetterInterface $template): string
    {
        return $template->getSite()->getName();
    }

    /**
     * @param EntityInterface $entity
     * @return string
     */
    public function formatActions(EntityInterface $entity): string
    {
        return $this->controlBuilder->buildActionMenuAdmin([], $entity->getId())->render();
    }

}