<?php

namespace DiamondTable\Commands;


use ActiveTable\ColumnTable;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use ActiveTable\FormField;
use Core\Form\Control\Hidden;
use Core\Form\Control\Button;
use Core\Form\Control\Submit;
use Core\Form\Control\FormControl;
use Core\Form\Simpleform;
use DiamondTable\ValidatorFactory;
use Propel\Runtime\Map\TableMap;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Validator\Validation;
use Twig\Environment;
use Twig\Loader\FilesystemLoader;

/**
 * отображение формы
 * Class FormView
 * @package ActiveTable\Commands
 */
class FormView implements CommandInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    protected $engine;

    public function __construct(DataTableEngine $tableEngine)
    {
        $this->engine = $tableEngine;
        $this->logger = new NullLogger();
    }

    /**
     *
     */
    public function process(): void
    {
        $form = new Simpleform($this->engine->getName());
        $form->setTwig(new Environment(new FilesystemLoader()));
        $form->setAction(adaptive_url([]));
        $form->setClass($this->engine->getFormClass());

        $form->addValidationSuccessEvent([$this, 'validationSuccess']);
        $form->addValidationPrepareEvent([$this, 'validationProcess']);

        $rowData = [];

        $queryParams = $this->engine->getRequest()->getQueryParams();

        $id = $this->engine->getCriteria()->getfilterById();

        //редактирование
        if(isset($queryParams['fn']) && $queryParams['fn']==='edit' && $id){
            $entity = $this->engine->getRepo()->findById($id);
            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $form->addHeader('Редактирование записи №' . $id);
            $form
                ->addField(
                    new  Hidden('id')
                );
        }
        //копирование
        elseif(isset($queryParams['fn']) && $queryParams['fn']==='add' && $id){
            $entity = $this->engine->getRepo()->findById($id);
            $rowData = $entity->toArray();
            $rowData = array_change_key_case($rowData);
            $form->addHeader('Новая запись');
        }
        //новая
        else{
            $entity = null;
            $form->addHeader('Новая запись');
        }


        $postData = $this->engine->getRequest()->getParsedBody();


        /**
         * @var FormField $formControl
         * @var FormControl $control
         * @var ColumnTable $column
         */
        foreach ($this->engine->getFields() as $formControl) {

            $control = $formControl->getControl();
            $name = strtolower($control->getName());

            $caption = $this->getCaption($formControl);

            if (is_a($control, \Core\Form\Control\FormField::class)) {

                if(isset($postData[$name])){
                    $control->setValue($postData[$name]);
                }
                elseif($entity && (is_callable($control->getFn()))){
                    $control->setValue(call_user_func($control->getFn(), $entity));
                }
                elseif (isset($rowData[$name])) {
                    $control->setValue($rowData[$name]);
                }

                $form->addField(
                    $control,
                    $caption,
                    $formControl->isRequire() === true ? 'required' : '',
                    $formControl->getHelpCaption()
                );
            } else {
                $form->addControl($control);
            }
        }


        if($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_SAVE_BUTTON)){
            $form
                ->addControl(
                (new Submit('submit', 'Сохранить'))
                    ->setClass('stdbtn btn_orange')
            );
        }

        if($this->engine->hasControlAccess(DataTableEngine::CONTROL_FORM_CANCEL_BUTTON)){

            $form->addControl(
                (new Button('cancel', 'Отмена'))
                    ->setType('button')
                    ->setClass('stdbtn')
                    ->setOnClick("location.href='" . adaptive_url([], ['fn', 'id']) . "'")
            );
        }


        $this->engine->addContent($form->render() );
    }

    /**
     * @param FormField $formControl
     * @return string|null
     */
    private function getCaption(FormField $formControl): ?string
    {
        $captions = [];
        $control = $formControl->getControl();
        $caption = null;

        foreach ($this->engine->getColumns() as $column) {
            $captions[$column->getName()] = $column->getCaption();
        }

        $name = $control->getName();
        if (!$caption = $formControl->getCaption()) {
            $caption = $captions[$name] ?? '';
        }

        return $caption;
    }

    /**
     * @param array $data
     * @param Simpleform $form
     * @return array|null
     */
    public function validationProcess(array $data, Simpleform $form): ?array
    {

        $validator = Validation::createValidator();
        $check = true;

        /**
         * @var FormField $field
         */
        foreach ($this->engine->getFields() as $field) {

            $name = $field->getControl()->getName();

            $rules = ValidatorFactory::build($field->isRequire() === true ? 'required' : '');

            $results = $validator->validate($data[$name] ?? null, $rules);

            foreach ($results as $violation) {
                $form->addCustomMessage(
                    $this->getCaption($field) . ' - ' . $violation->getMessage(), Simpleform::MESSAGE_ERROR
                );
                $check = false;
            }

        }

        return $check === true ? $data : null;
    }

    /**
     * @param array $data
     * @param Simpleform $form
     */
    public function validationSuccess(array $data, Simpleform $form):void
    {

        /**
         * @var ActiveRecordInterface $entity
         */
        $id = $data['id'] ?? null;
        unset($data['id']);

        //create
        if ($id) {
            $entity = $this->engine->getRepo()->findById($id);
            $entity->fromArray($data, TableMap::TYPE_CAMELNAME);
        } //update
        else {
            $entity = $this->engine->getRepo()->buildEntityFromArray($data);
        }

        try {
            $this->engine->getRepo()->save($entity);
            $session = new Session();
            $session->getFlashBag()->set('table-notice',$form->createMessage(
                'Запись №'.$entity->getId().' успешно сохранена! ' . anchor(adaptive_url([]),'Добавить новую запись'
                ),
                Simpleform::MESSAGE_SUCCESS
            ));
            RedirectResponse::create(adaptive_url([], ['fn','id']))->send();
        } catch (\Exception $ex) {
            $this->logger->error($ex->getMessage());
            $newForm = clone $form;
            $form->addCustomMessage('При сохранении возникла ошибка', Simpleform::MESSAGE_ERROR);
            $form->addBuffer($newForm->generateForm());
        }
    }
}