<?php

namespace DiamondTable;

use ActiveTable\Contracts\CommandFactoryInterface;
use ActiveTable\Contracts\CommandInterface;
use ActiveTable\DataTableEngine;
use ActiveTable\Exceptions\ActiveTableException;
use DiamondTable\Commands\FormView;
use DiamondTable\Commands\TableAction;
use DiamondTable\Commands\TableView;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Фабрика комманд
 * Class CommandFactory
 * @package ActiveTable\Factories
 */
final class CommandFactory implements CommandFactoryInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    public const TABLE_VIEW = 'TABLE_VIEW';
    public const FORM_VIEW = 'FORM_VIEW';
    public const TABLE_ACTION = 'TABLE_ACTION';

    protected $events = [
        self::TABLE_VIEW,
        self::FORM_VIEW,
        self::TABLE_ACTION
    ];

    /**
     * @var string
     */
    protected $event;

    /**
     * @return string
     */
    public function getEventName(): string
    {
        return $this->event;
    }


    /**
     * @param ServerRequestInterface $request
     * @return CommandFactory
     * @throws ActiveTableException
     */
    public static function buildFormRequest(ServerRequestInterface $request): self
    {

        $postData = $request->getParsedBody();
        $getData = $request->getQueryParams();

        if (
            isset($postData['actiontype']) ||
            (isset($getData['fn']) && in_array($getData['fn'], ['delete', 'exportTable']))
        ) {
            return new self(self::TABLE_ACTION);
        } elseif (
            isset($postData['submit'])||
            (isset($getData['fn']) && in_array($getData['fn'], ['add', 'edit']))
        ) {
            return new self(self::FORM_VIEW);
        }


        return new self();
    }

    /**
     * CommandFactory constructor.
     * @param $event
     * @throws ActiveTableException
     */
    public function __construct(string $event = self::TABLE_VIEW)
    {
        if (!in_array($event, $this->events, true)) {
            throw new ActiveTableException('event not found');
        }
        $this->event = $event;
        $this->logger = new NullLogger();
    }

    /**
     * Возвращаем комманды в зависимости от ситуации
     * @param DataTableEngine $tableEngine
     * @return CommandInterface
     */
    public function build(DataTableEngine $tableEngine): CommandInterface
    {
        switch ($this->event) {
            case self::TABLE_VIEW;

                //стандартное отображение таблицы
                return new TableView(
                    $tableEngine
                );

                break;

            case self::FORM_VIEW;

                //стандартное отображение таблицы
                $form = new FormView(
                    $tableEngine
                );
                $form->setLogger($this->logger);
                return $form;

                break;

            case self::TABLE_ACTION;

                //стандартное отображение таблицы
                return new TableAction($tableEngine);

                break;
        }
    }

}