<?php declare(strict_types=1);

namespace Diamond\Modules\AdminMailingRecipientsList;


use ActiveTable\ColumnTable;
use ActiveTable\DataTableEngine;
use Core\Form\ControlBuilder;
use Core\Helpers\Date;
use Core\ModuleController;
use Diamond\Application\Mailing\MailingService;
use Diamond\Domain\MailingRequestRecipient\Contracts\MailingRequestRecipientEntityInterface;
use Diamond\Domain\MailingSubscriber\Contracts\MailingSubscriberEntityInterface;
use Diamond\Repositories\MailingRequestRecipient\MailingRequestRecipientCriteria;
use DiamondTable\TableFactory;

class AdminMailingRecipientsList extends ModuleController
{

    public function __construct(
        private MailingService $mailingService,
        private TableFactory   $tableFactory,
        private ControlBuilder $controlBuilder
    )
    {}

    public function process(): string
    {

        if (!$request = (int)input_get('request_id')) {
            return '';
        }

        $table = $this->tableFactory->buildAdminTable(
            $this->mailingService->getMailingRequestRecipientRepository(),
            MailingRequestRecipientCriteria::create()->setFilterByRequestId($request)
        );

        $table
            ->setDefaultSortColumn('id', 'desc')
            ->removeControlAccess(DataTableEngine::CONTROL_ACCESS_ADD)
            ->removeControlAccess(DataTableEngine::CONTROL_ROWS_ACTION)
            ->removeControlAccess(DataTableEngine::CONTROL_ACCESS_DELETE)
            ->addColumn(
                (new ColumnTable('id', '№'))
                    ->setWidth(40)
            )
            ->addColumn(
                (new ColumnTable('email', 'Email'))
            )
            ->addColumn(
                (new ColumnTable('created_at', 'Дата создания'))->setFormat($this, 'formatDate1')
            )
            ->addColumn(
                (new ColumnTable('sending_at', 'Дата отправки'))->setFormat($this, 'formatDate2')
            )
            ->addColumn(
                (new ColumnTable('viewed_at', 'Дата просмотра'))->setFormat($this, 'formatDate3')
            );

        return $table->render();
    }

    public function formatDate1( MailingRequestRecipientEntityInterface $stat): string
    {
        return $stat->getCreatedAt() ? $stat->getCreatedAt(Date::formatDisplay()) : '';
    }

    public function formatDate2(MailingRequestRecipientEntityInterface $stat): string
    {
        return $stat->getSendingAt() ? $stat->getSendingAt(Date::formatDisplay()) : '';
    }

    public function formatDate3(MailingRequestRecipientEntityInterface $stat): string
    {
        return $stat->getViewedAt() ? $stat->getViewedAt(Date::formatDisplay()) : '';
    }
}