<?php


namespace Diamond\Application\Mailing;

use Core\MailingPreparation\MailingCook;
use Diamond\Domain\MailingList\Concrete\MailingListEntityInterface;
use Diamond\Domain\MailingList\MailingListCollection;
use Diamond\Application\Mailing\Exceptions\MailingException;
use Diamond\Application\Mailing\Exceptions\MailingListNotFoundException;
use Diamond\Infrastructure\Models\MailingRequest\MailingRequest;
use Diamond\Infrastructure\Models\MailingRequestRecipient\MailingRequestRecipient;
use Diamond\Repositories\MailingList\MailingListCriteria;
use Diamond\Repositories\MailingList\MailingListRepository;
use Diamond\Repositories\MailingRequestRecipient\MailingRequestRecipientRepository;
use Diamond\Repositories\MailingSubscriber\MailingSubscriberRepository;
use Diamond\Repositories\MailingTemplate\MailingTemplateRepository;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Бизнес процессы рассылки
 * Class Mailing
 * @package Diamond\UseCases\Mailing
 */
class MailingService implements LoggerAwareInterface
{
    use LoggerAwareTrait;


    public function __construct(private MailingListRepository       $mailingListRepository,
                                private MailingSubscriberRepository $mailingSubscriberRepository,
                                private MailingRequestRecipientRepository $mailingRequestRecipientRepository,
                                private MailingTemplateRepository   $mailingTemplateRepository,
                                private MailingCook                 $mailingCook,
                                private MailingRecipientsCollection $mailingRecipientsCollection
    )
    {
    }

    /**
     * @return MailingRequestRecipientRepository
     */
    public function getMailingRequestRecipientRepository(): MailingRequestRecipientRepository
    {
        return $this->mailingRequestRecipientRepository;
    }


    /**
     * @return MailingListRepository
     */
    public function getMailingListRepository(): MailingListRepository
    {
        return $this->mailingListRepository;
    }

    /**
     * @return MailingSubscriberRepository
     */
    public function getMailingSubscriberRepository(): MailingSubscriberRepository
    {
        return $this->mailingSubscriberRepository;
    }

    /**
     * @return MailingTemplateRepository
     */
    public function getMailingTemplateRepository(): MailingTemplateRepository
    {
        return $this->mailingTemplateRepository;
    }


    /**
     * получение листа рассылки
     * @param int $id
     * @return MailingListEntityInterface
     */
    public function getMailingList(int $id): MailingListEntityInterface
    {
        if (!$mailing = $this->mailingListRepository->findById($id)) {
            throw new MailingListNotFoundException(sprintf(
                'mailing list not found by id `%s`', $id
            ));
        }
        return $mailing;
    }

    /**
     * @param MailingListEntityInterface $mailingListEntity
     * @param string|null $email
     * @param string|null $key
     * @return string
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\SyntaxError
     */
    public function getMailingListBodyForSubscriber(
        MailingListEntityInterface $mailingListEntity,
        string                     $email = null,
        string                     $key = null
    ): string
    {
        $key = $key ?? md5($email);

        $subscribeEmails = $this->getSubscribeEmails($mailingListEntity);
        $email = $email ?? array_pop($subscribeEmails);

        $body = $mailingListEntity->getMailingTemplate()->getBody();

        $body = $this->mailingCook->prepare($body, $key);

        return $body;
    }

    /**
     * @param MailingListEntityInterface $mailingListEntity
     * @return array
     * @todo нужна работа драйвеов по получению подписчиков
     */
    public function getSubscribeEmails(MailingListEntityInterface $mailingListEntity): array
    {
        $emails = [];

        $sub = $mailingListEntity->getMailingSubscriber();


        foreach ($this->mailingRecipientsCollection->getHandlers() as $mailingRecipients) {
            if (get_class($mailingRecipients) === $sub->getFn()) {
                $emails = $mailingRecipients->getMails();
            }
        }

        $_emails = explode(';', $sub->getCustomMails());

        $_emails = array_merge($emails, $_emails);

        $_ignore = explode(';', $sub->getIgnoreMails());

        return \array_diff($_emails, $_ignore);
    }


    /**
     * Запуск рассылки, перевод в рабочее состояние
     * @todo в методе не работает блок $recipients, возможно рудемент??
     * @param MailingListEntityInterface $mailingListEntity
     */
    public function sendMailing(MailingListEntityInterface $mailingListEntity): void
    {

        $template = $mailingListEntity->getMailingTemplate();

        $body = $template->getBody();
        $subject = $template->getSubject();


        try {

            $request = (new MailingRequest())
                ->setBody($body)
                ->setSubject($subject)
                ->setMailingId($mailingListEntity->getId());

            $emails = $this->getSubscribeEmails($mailingListEntity);
            $emailsTo = [];

            if (!count($emails)) {
                throw new MailingException('receipts not set');
            }

            foreach ($emails as $mail) {
                $recipients = (new MailingRequestRecipient())
                    ->setMailingRequest($request)
                    ->setEmail($mail);
            }

            $request->save();

        } catch (\RuntimeException $exception) {
            $this->logger->error($exception->getMessage());
            throw new MailingException('send mailing error');
        }
    }


    /**
     * @return MailingListCollection
     * @throws \Repo\Concrete\Exceptions\Collection
     */
    public function getActiveMailing(): MailingListCollection
    {
        return $this->mailingListRepository->findByCriteria(
            MailingListCriteria::create()->setFilterByActive(true)
        );
    }

    /**
     * @return array
     * Формировать динамически список вариантов подписки
     */
    public function getRules()
    {
        $rules = [
            'custom' => 'любые адреса',
        ];

        $items = $this->mailingRecipientsCollection->getHandlers();

        foreach ($items as $item) {
            $rules[get_class($item)] = $item->getName();
        }

        return $rules;

//        return [
//            'allUsers'      => 'Система / Все пользователи',
//            'allClients'      => 'Магазин / Все клиенты',
////            'group3ClientsOffice1'      => 'Магазин / 3-я Группа клиентов, офис 1',
//            'blockClients'    => 'Магазин / Заблокированные клиенты',
//            'noneAuthClients' => 'Магазин / Неавторизованные клиенты',
//            'AuthClients'     => 'Магазин / Авторизованные клиенты',
//            'custom' => 'любые адреса',
//        ];
    }
}