<?php

/**
 * Created by PhpStorm.
 * User: d.lanec
 * Date: 03.08.2020
 * Time: 16:45
 */

namespace ExportEngine\Drivers;

use ExportEngine\Contracts\ExportDriverInterface;
use ExportEngine\Exceptions\ExportException;

class CsvLegacy extends AbstractDriver implements ExportDriverInterface
{
    protected $file;
    protected $data;
    protected $separator = ';';
    protected $enclosure = '"';
    protected $escape = '\\';
    protected $linebreak = PHP_EOL;
    protected $fileName;
    protected $charSet;
    protected $head;
    protected $fp;
    protected $ext = 'csv';
    protected $asFormulas = false;
    protected $strongEnclosure = false;


    /**
     * Csv constructor.
     * @param \SplFileObject $file
     */
    public function __construct(\SplFileObject $file)
    {
        $this->file = $file;
    }

    /**
     * @param bool $strongEnclosure
     * @return $this
     */
    public function setStrongEnclosure(bool $strongEnclosure)
    {
        $this->strongEnclosure = $strongEnclosure;
        return $this;
    }


    /**
     * Оичтка значения + кодировка
     * @param string $value
     * @return string
     */
    public function prepareValue(string $value): string
    {
        if ($this->charSet === 'windows-1251') {
            $value = iconv('utf-8', $this->charSet . '//TRANSLIT//IGNORE', $value);
        }
        return trim($value);
    }


    /**
     * @param bool $val
     * @return $this
     */
    public function setAsFormulas(bool $val)
    {
        $this->asFormulas =  $val;
        $this->strongEnclosure = true;
        return $this;
    }


    /**
     * @param array $head
     * @return $this
     */
    public function setHead(array $head)
    {
        $this->head = $head;
        return $this;
    }

    /**
     * @param string $charSet
     * @return $this
     */
    public function setCharSet(string $charSet)
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @return string
     */
    public function getExtendsion(): string
    {
        return $this->ext;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setFileName(string $name)
    {
        $this->fileName = $name;
        return $this;
    }

    /**
     * @param string $separator
     * @return $this
     */
    public function setSeparator(string $separator)
    {
        $this->separator = $separator;
        return $this;
    }

    /**
     * @param string $enclosure
     * @return $this
     */
    public function setEnclosure(string $enclosure)
    {
        $this->enclosure = $enclosure;
        return $this;
    }

    /**
     * @param string $escape
     * @return $this
     */
    public function setEscape(string $escape)
    {
        $this->escape = $escape;
        return $this;
    }

    /**
     * @param string $linebreak
     * @return $this
     */
    public function setLinebreak(string $linebreak)
    {
        $this->linebreak = $linebreak;
        return $this;
    }

    /**
     * @param string $fileName
     */
    public function output(string $fileName): void
    {
        $filename = filter_var($this->fileName, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
        header('Content-Transfer-Encoding: binary');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Content-Type: text/csv; charset=' . $this->charSet);
        $this->outputOnlyData();
    }

    /**
     * вывод чистых данных в буфер без заголовка
     */
    protected function outputOnlyData(): void
    {
        echo $this->getFirstLine();

        foreach ($this->readTransport->read() as $row) {
            foreach ($row as $id => $value) {
                $row[$id] = $this->prepareValue((string)$value);
            }
            echo $this->pepareRow($row);
        }

        echo $this->getLastLine();

        $this->readTransport->close();
    }

    /**
     * @return string
     */
    protected function getFirstLine(): string
    {
        if (count($this->head)) {
            return $this->makeCsv($this->head);
        }

        return '';
    }

    /**
     * обработка строки массива
     * @param array $row
     * @return string
     */
    protected function pepareRow(array $row): string
    {
        return $this->makeCsv($row);
    }

    /**
     * @param array $rows
     * @return string
     */
    private function makeCsv(array $rows): string
    {
        $csv = '';

        array_walk($rows, function (&$value) {
            $value = trim($value);
        });
        $form = $this->asFormulas === true ? '=' : '';

        foreach ($rows as $row) {
            $value = $this->prepareValue((string)$row);
            $enc = $this->enclosure;
            if ($this->strongEnclosure === false && strpos($value, ' ') === false) {
                $enc = '';
            }
            $csv .= $form . $enc . $value . $enc . $this->separator;
        }

        return rtrim($csv, $this->separator) . $this->linebreak;
    }


    /**
     * @return string
     */
    protected function getLastLine(): string
    {
        return '';
    }

    /**
     * @param string $path
     * @throws ExportException
     */
    public function exportToFile(string $path): void
    {
        if (!$outputHandle = fopen($path, 'w+')) {
            throw new ExportException('Unable to open file to write!');
        }

        fwrite($outputHandle, $this->getFirstLine());

        //$this->readTransport->clearPointer();

        foreach ($this->readTransport->read() as $row) {
            fwrite($outputHandle, $this->pepareRow($row));
        }

        $this->readTransport->close();

        fwrite($outputHandle, $this->getLastLine());

        fflush($outputHandle);

        fclose($outputHandle);
    }

    /**
     * @param array $data
     * @return $this
     */
    public function setData(array $data)
    {
        $this->data = $data;
        return $this;
    }
}
