<?php

namespace ExportEngine\Factories;

use ExportEngine\Contracts\ExportDriverInterface;
use ExportEngine\Exceptions;
use ExportEngine\Drivers;

/**
 * Фабрика драйверов
 *
 * @author d.lanec
 */
class DriverFactory
{

    /**
     * Создание объекта по типу
     * @param string $type
     * @param bool $csvLegacy
     * @return ExportDriverInterface
     * @throws Exceptions\FormatNotSupported
     */
    public static function buildByType(string $type, bool $csvLegacy = true): ExportDriverInterface
    {

        if ($csvLegacy === true && in_array($type, ['csv', 'txt'])) {
            $type .= 'Legacy';
        }

        if (!method_exists(self::class, 'build' . $type)) {
            throw new Exceptions\FormatNotSupported($type);
        }

        return self::{'build' . $type}();
    }


    /**
     * @return Drivers\CsvLegacy
     */
    public static function buildCsvLegacy(): Drivers\CsvLegacy
    {
        return new Drivers\CsvLegacy(new \SplTempFileObject());
    }

    /**
     * @return Drivers\TxtLegacy
     */
    public static function buildTxtLegacy(): Drivers\TxtLegacy
    {
        return new Drivers\TxtLegacy(new \SplTempFileObject());
    }

    /**
     * @return Drivers\Csv
     */
    public static function buildCsv(): Drivers\Csv
    {
        return new Drivers\Csv();
    }

    /**
     * @return Drivers\Txt
     */
    public static function buildTxt(): Drivers\Txt
    {
        return new Drivers\Txt();
    }

    public static function buildJson(): Drivers\JsonDriver
    {
        return new Drivers\JsonDriver();
    }

    /**
     * @return Drivers\Xml
     */
    public static function buildXML(): Drivers\Xml
    {
        return new Drivers\Xml();
    }

    /**
     * @return Drivers\Xml2
     */
    public static function buildXML2(): Drivers\Xml2
    {
        return new Drivers\Xml2();
    }

    public static function buildXMLDrom(): Drivers\Xml2
    {
        return new Drivers\XmlDrom();
    }

    /**
     * @return Drivers\Xlsx
     */
    public static function buildXlsx(): Drivers\Xlsx
    {
        return new Drivers\Xlsx();
    }

    /**
     * @return Drivers\Xls
     */
    public static function buildXls(): Drivers\Xls
    {
        return new Drivers\Xls();
    }

    /**
     * @return Drivers\Html
     */
    public static function buildHtml(): Drivers\Html
    {
        return new Drivers\Html();
    }

    /**
     * @return Drivers\YandexMarket
     */
    public static function buildYml(): Drivers\YandexMarket
    {
        return new Drivers\YandexMarket();
    }
}
