<?php

namespace ExportEngine\Drivers;

use ExportEngine\Contracts\ExportDriverInterface;

class YandexMarket extends AbstractDriver implements ExportDriverInterface
{
    protected $organizationName;
    protected $organizationFullName;
    protected $siteUrl;
    protected $currencyList;
    protected $deliveryCost;
    protected $deliveryDays;
    protected $deliveryOrderBefore;

    protected $categoryList;
    protected $lastCategoryId = 0;
    protected $lastOfferId = 0;

    public function output(string $fileName): void
    {
        // TODO: Implement output() method.
    }

    public function exportToFile(string $path): void
    {
        if (!$outputHandle = fopen($path, 'w+')) {
            throw new ExportException('Unable to open file to write!');
        }

        $this->makeCategoryList();

        //fwrite($outputHandle, $this->getFirstLine());
        fwrite($outputHandle, $this->getHeaderBlock());

        $this->makeCategoryBlock($outputHandle);

        $this->readTransport->rewind();

        $this->makeOfferBlock($outputHandle);

        fwrite($outputHandle, PHP_EOL . $this->getFooterBlock());

        fflush($outputHandle);

        fclose($outputHandle);

        $this->readTransport->close();

        //$writer->close();
    }

    protected function getCategoryKey($categoryLine)
    {
        return md5($categoryLine);
    }

    /**
     * @param $categoryLine
     * @return __anonymous@1500
     */
    protected function getCategoryObject($categoryLine)
    {
        $key = $this->getCategoryKey($categoryLine);
        if (isset($this->categoryList[$key])) {
            return $this->categoryList[$key];
        }

        $result = new class {
            public $id;
            public $title;
            public $parentId;
        };
        if (empty($categoryLine)) {
            $result->id = 0;
            $result->title = '';
        } else {
            $list = explode("|", $categoryLine);
            $result->title = array_pop($list);
            $parentObject = $this->getCategoryObject(implode("|", $list));
            $result->parentId = $parentObject->id;
            $result->id = ++$this->lastCategoryId;
        }
        $this->categoryList[$key] = $result;
        return $result;
    }

    protected function makeCategoryList()
    {
        $i = 0;
        foreach ($this->readTransport->read() as $row) {
            if (!isset($row['categoryLine'])) {
                continue;
            }
            $this->getCategoryObject($row['categoryLine']);
        }
    }

    protected function makeCategoryBlock($outputHandle)
    {
        if (is_array($this->categoryList)) {
            fwrite($outputHandle, PHP_EOL . "<categories>");
            foreach ($this->categoryList as $categoryObject) {
                if ($str = $this->prepareCategoryXml($categoryObject)) {
                    fwrite($outputHandle, PHP_EOL . $str);
                }
            }
            fwrite($outputHandle, PHP_EOL . "</categories>");
        }
    }

    protected function prepareCategoryXml($categoryObject)
    {
        $attrList = [];

        if (empty($categoryObject->id)) {
            return '';
        }

        $attrList['id'] = $categoryObject->id;
        if (!empty($categoryObject->parentId)) {
            $attrList['parentId'] = $categoryObject->parentId;
        }
        return $this->getElementXml('category', $categoryObject->title, $attrList, true);
    }

    protected function makeOfferBlock($outputHandle)
    {
        fwrite($outputHandle, PHP_EOL . "<offers>");
        foreach ($this->readTransport->read() as $row) {
            fwrite($outputHandle, PHP_EOL . $this->prepareOfferXml($row));
        }
        fwrite($outputHandle, PHP_EOL . "</offers>");
    }

    protected function prepareOfferXml($data)
    {
        $otherData = $data['otherData'] ?? [];
        $url = $otherData['url'] ?? '';


//        if (empty($data['price']) || empty($data['product_name']) || empty($url)) {
//            return '';
//        }
        $list = [];
        $list[] = '<offer id="' . ++$this->lastOfferId . '">';

        if (isset($data['url'])) {
            $list[] = $this->getElementXml('url', $data['url']);
        }

        if (isset($data['price'])) {
            $list[] = $this->getElementXml('price', $data['price'], [], true);
        }


        $list[] = $this->getElementXml('currencyId', $this->getMainCurrencyIso(), [], true);

        if (isset($data['categoryId'])) {
            $list[] = $this->getElementXml('categoryId', $this->getCategoryId($data['categoryLine']), [], true);
        }


        if (!empty($data['image'])) {
            $list[] = $this->getElementXml('picture', $data['image'], [], true);
        }

        if (!empty($data['producerName'])) {
            $list[] = $this->getElementXml('vendor', $data['producerName'], [], true);
        }

        if (!empty($data['code'])) {
            $list[] = $this->getElementXml('vendorCode', $data['code'], [], true);
        }
        if (!empty($data['product_name'])) {
            $list[] = $this->getElementXml('product_name', $data['product_name'], [], true);
        }

        if (!empty($data['description'])) {
            $list[] = $this->getElementXml('description', '<![CDATA[' . $data['description'] . ']]>', []);
        }

        $propertyList = $otherData['propertyList'] ?? [];
        foreach ((array)$propertyList as $paramName => $paramValue) {
            if (!is_scalar($paramValue)) {
                continue;
            }
            $list[] = $this->getElementXml('param', $paramValue, ['name' => $paramName], true);
        }

        $list[] = $this->getElementXml('url', $url, [], true);

        $list[] = "</offer>";
        return implode(PHP_EOL, $list);
    }

    protected function getMainCurrencyIso()
    {
        //TODO: откуда брать валюту
        return 'RUB';
    }

    protected function getCategoryId($categoryLine)
    {

        return $this->getCategoryObject($categoryLine)->id;
    }


    /**
     * @param string $organizationName
     * @return YandexMarket
     */
    public function setOrganizationName(string $organizationName)
    {
        $this->organizationName = $organizationName;
        return $this;
    }

    /**
     * @param string $organizationFullName
     * @return YandexMarket
     */
    public function setOrganizationFullName(string $organizationFullName)
    {
        $this->organizationFullName = $organizationFullName;
        return $this;
    }

    /**
     * @param string $siteUrl
     * @return YandexMarket
     */
    public function setSiteUrl(string $siteUrl)
    {
        $this->siteUrl = $siteUrl;
        return $this;
    }

    /**
     * @param array $currencyList
     * @return YandexMarket
     */
    public function setCurrencyList(array $currencyList)
    {
        $this->currencyList = $currencyList;
        return $this;
    }

    /**
     * @param mixed $deliveryCost
     * @return YandexMarket
     */
    public function setDeliveryCost($deliveryCost)
    {
        $this->deliveryCost = $deliveryCost;
        return $this;
    }

    /**
     * @param mixed $deliveryDays
     * @return YandexMarket
     */
    public function setDeliveryDays($deliveryDays)
    {
        $this->deliveryDays = $deliveryDays;
        return $this;
    }

    /**
     * @param mixed $deliveryOrderBefore
     * @return YandexMarket
     */
    public function setDeliveryOrderBefore($deliveryOrderBefore)
    {
        $this->deliveryOrderBefore = $deliveryOrderBefore;
        return $this;
    }

    /**
     * @return string
     */
    protected function getHeaderBlock(): string
    {
        $res = '<?xml version="1.0" encoding="UTF-8"?><!DOCTYPE yml_catalog SYSTEM "shops.dtd">' . PHP_EOL;
        $res .= '<yml_catalog date="' . date('Y-m-d H:i') . '"><shop>' . PHP_EOL;
        $res .= $this->getElementXml("name", $this->organizationName, [], true) . PHP_EOL;
        $res .= $this->getElementXml("company", $this->organizationFullName, [], true) . PHP_EOL;
        $res .= $this->getElementXml("url", $this->siteUrl, [], true) . PHP_EOL;

        $res .= $this->getCurrenciesXml() . PHP_EOL;

        $deliveryOptions = [];
        if (!empty($this->deliveryCost)) {
            $deliveryOptions['cost'] = $this->deliveryCost;
        }
        if (!empty($this->deliveryDays)) {
            $deliveryOptions['days'] = $this->deliveryDays;
        }
        $orderBefore = (int)$this->deliveryOrderBefore;
        if (($orderBefore > 0) && ($orderBefore < 24)) {
            $deliveryOptions['order-before'] = $orderBefore;
        }
        if (!empty($deliveryOptions)) {
            $optionsXml = PHP_EOL . $this->getElementXml('option ', '', $deliveryOptions) . PHP_EOL;
            $res .= $this->getElementXml('delivery-options', $optionsXml) . PHP_EOL;
        }

        return $res;
    }

    public function getCurrenciesXml()
    {
        $xml = '';

        if (is_array($this->currencyList)) {
            $xml .= "<currencies>" . PHP_EOL;
            foreach ($this->currencyList as $curCode => $rate) {
                $xml .= $this->getElementXml('currency', '', ['id' => $curCode, 'rate' => $rate]) . PHP_EOL;
            }
            $xml .= "</currencies>";
        }

        return $xml;
    }

    protected function getFooterBlock()
    {
        return '</shop></yml_catalog>';
    }

    public function getElementXml($name, $content, $attr = [], $replaceContentHtmlEntities = false)
    {

        $attrStrArr = [];
        foreach ((array)$attr as $key => $value) {
            $attrStrArr[] = $key . '="' . $this->replaceHtmlEntities($value) . '"';
        }

        $str = '<' . $name . (empty($attrStrArr) ? '' : ' ' . implode(' ', $attrStrArr)) . '>';
        if ($replaceContentHtmlEntities) {
            $content = $this->replaceHtmlEntities($content);
        }

        $str .= $content;
        $str .= '</' . $name . '>';

        return $str;
    }

    public function replaceHtmlEntities($text)
    {

        return str_replace(array("&", "'", "\"", ">", "<"), array("&amp;", "&apos;", "&quot;", "&gt;", "&lt;"), $text);
    }
}
