<?php

namespace ExportEngine\Drivers;

use Box\Spout\Common\Exception\IOException;
use Box\Spout\Writer\Common\Creator\WriterEntityFactory;
use Box\Spout\Writer\WriterFactory;
use ExportEngine\Contracts\ExportDriverInterface;
use ExportEngine\Exceptions\ExportException;

/**
 * Csv export driver
 */
class Csv extends AbstractDriver implements ExportDriverInterface
{
    protected $ext = 'csv';
    protected $separator = ';';
    protected $enclosure = '"';
    protected $strongEnclosure = false;

    /**
     * @param string $separator
     * @return $this
     */
    public function setSeparator(string $separator): self
    {
        $this->separator = $separator;
        return $this;
    }

    /**
     * @param string $enclosure
     * @return $this
     */
    public function setEnclosure(string $enclosure): self
    {
        $this->enclosure = $enclosure;
        return $this;
    }

    protected function setOptions($writer, $ext)
    {
        if ($ext === 'csv') {
            if (!empty($this->separator)) {
                $writer->setFieldDelimiter($this->separator);
            }

            if (!empty($this->enclosure)) {
                $writer->setFieldEnclosure($this->enclosure);
            }

            $writer->setShouldAddBOM(false);
        }
    }

    /**
     * @param string $fileName
     * @throws IOException
     * @throws \Box\Spout\Common\Exception\UnsupportedTypeException
     * @throws \Box\Spout\Writer\Exception\WriterNotOpenedException
     */
    public function output(string $fileName): void
    {
        $ext = $this->getExtension();
        $writer = WriterEntityFactory::createWriter($ext);
        $this->setOptions($writer, $ext);

        $writer->openToBrowser($fileName);

        if(count($this->head)){
            $writer->addRow(WriterEntityFactory::createRowFromArray($this->head));
        }


        foreach ($this->readTransport->read() as $row) {
            foreach ($row as $id => $value) {
                $row[$id] = $this->prepareValue($value);
            }
            $writer->addRow(WriterEntityFactory::createRowFromArray($row));
        }

        $this->readTransport->close();

        $writer->close();
    }

    /**
     * @param string $path
     * @throws ExportException
     * @throws IOException
     * @throws \Box\Spout\Common\Exception\UnsupportedTypeException
     * @throws \Box\Spout\Writer\Exception\WriterNotOpenedException
     */
    public function exportToFile(string $path): void
    {

        $ext = $this->getExtension();
        $writer = WriterEntityFactory::createWriter($ext);

        try {
            $writer->openToFile($path);
        } catch (IOException $e) {
            throw new ExportException('Unable to open file to write! error: ' . $e->getMessage());
        }

        if(count($this->head)){
            $writer->addRow(WriterEntityFactory::createRowFromArray($this->head));
        }

        foreach ($this->readTransport->read() as $row) {
            foreach ($row as $id => $value) {
                $row[$id] = $this->prepareValue($value);
            }
            $writer->addRow(WriterEntityFactory::createRowFromArray($row));
        }

        $this->readTransport->close();

        $writer->close();
    }
}
