<?php

namespace ExportEngine;

use ExportEngine\Contracts\ExporterInterface;
use ExportEngine\Service\Serialize\WriteTransport;

class Exporter implements ExporterInterface
{
    protected $driver;

    protected $tmpFile;
    protected $writeTransport;

    /**
     * @param Contracts\ExportDriverInterface $driver
     */
    public function __construct(Contracts\ExportDriverInterface $driver)
    {
        $this->driver = $driver;

        $this->tmpFile = tempnam(sys_get_temp_dir(), 'data-export-');
        $this->writeTransport = new WriteTransport($this->tmpFile);
    }

    public function __destruct()
    {
        $this->writeTransport->close();
        unlink($this->tmpFile);
    }

    /**
     * @param array $header
     * @return Exporter
     */
    public function setHeader(array $header): self
    {
        $this->driver->setHead($header);
        return $this;
    }

    /**
     * @param array $row
     * @return Exporter
     */
    public function addRow(array $row): self
    {
        $this->writeTransport->write($row);
        return $this;
    }

    /**
     * @param array $data
     * @return Exporter
     */
    public function setData(array $data): self
    {
        foreach ($data as $row) {
            $this->writeTransport->write($row);
        }
        return $this;
    }

    /**
     * @param string $fileName
     */
    public function output(string $fileName): void
    {
        $this->writeTransport->close();

        $this
            ->driver
            ->setSourceData($this->tmpFile)
            ->output($fileName);
    }

    /**
     * @param string $path
     */
    public function exportToFile(string $path): void
    {
        $this->writeTransport->close();

        $this
            ->driver
            ->setSourceData($this->tmpFile)
            ->exportToFile($path);
    }
}
