<?php

namespace ExportEngine\Drivers;

use ExportEngine\Contracts\ExportDriverInterface;

/**
 *  Xml export driver
 */
class Xml extends AbstractDriver implements ExportDriverInterface
{
    protected $ext = 'xml';

    /**
     * @param string $fileName
     */
    public function output(string $fileName): void
    {
        header('Content-type: application/xml');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');

        $this->outputOnlyData();
    }

    protected function outputOnlyData(): void
    {
        echo $this->getFirstLine();

        foreach ($this->readTransport->read() as $row) {
            echo $this->prepareRow($row);
        }

        echo $this->getLastLine();
    }


    /**
     * @param string $path
     * @throws ExportException
     */
    public function exportToFile(string $path): void
    {
        if (!$outputHandle = fopen($path, 'w+')) {
            throw new ExportException('Unable to open file to write!');
        }

        fwrite($outputHandle, $this->getFirstLine());

        foreach ($this->readTransport->read() as $row) {
            fwrite($outputHandle, $this->prepareRow($row));
        }

        fwrite($outputHandle, $this->getLastLine());

        fflush($outputHandle);

        fclose($outputHandle);
    }

    /**
     * @param array $row
     * @return string
     */
    protected function prepareRow(array $row): string
    {
        $res = '<row>' . PHP_EOL;
        foreach ($row as $item) {
            $res .= '<cell>' . $this->prepareValue($item) . '</cell>' . PHP_EOL;
        }
        return $res . '</row>' . PHP_EOL;
    }

    /**
     * @return string
     */
    protected function getFirstLine(): string
    {
        $res = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $res .= '<rows>' . PHP_EOL;
        return $res;
    }

    /**
     * @return string
     */
    protected function getLastLine(): string
    {
        return '</rows>';
    }
}
