<?php

namespace ExportEngine;

use PHPUnit\Framework\TestCase;
use ExportEngine\Factories\DriverFactory;

class ExporterTest extends TestCase
{
    /**
     * @var Exporter
     */
    protected $object;

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutput()
    {
        $delimiter = ';';
        $enclosure = '"';

        $this->object = new Exporter(DriverFactory::buildCsv());
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @covers \ExportEngine\Exporter::output
     * @throws \Exception
     */
    public function testCsvLegacyOutputtestStrongEnclosure()
    {
        $delimiter = ';';
        $enclosure = '"';

        $this->object = new Exporter(DriverFactory::buildCsvLegacy()->setStrongEnclosure(true));
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure, true);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @covers \ExportEngine\Exporter::output
     * @throws \Exception
     */
    public function testCsvLegacyOutputtestAsFormulas()
    {
        $delimiter = ';';
        $enclosure = '"';

        $this->object = new Exporter(DriverFactory::buildCsvLegacy()->setAsFormulas(true));
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure, true, true);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutputSeparator()
    {
        $delimiter = ',';
        $enclosure = '"';

        $this->object = new Exporter(DriverFactory::buildCsv()->setSeparator($delimiter));
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutputEnclosure()
    {
        $delimiter = ';';
        $enclosure = "'";

        $this->object = new Exporter(DriverFactory::buildCsv()->setEnclosure($enclosure));
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testTxtOutputEnclosure()
    {
        $delimiter = ';';
        $enclosure = "'";

        $this->object = new Exporter(DriverFactory::buildTxt()->setEnclosure($enclosure));
        list($header, $data) = $this->createDataTests();

        $fields = array_merge([$header], $data);

        $result = $this->makeCsv($fields, $delimiter, $enclosure);

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testXmlOutput()
    {
        $this->object = new Exporter(DriverFactory::buildXML());
        list($header, $data) = $this->createDataTests();

        $result = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $result .= '<rows>' . PHP_EOL;
        $_data = $data;
        foreach ($_data as $row) {
            $result .= '<row>' . PHP_EOL .
                '<cell>' . implode('</cell>' . PHP_EOL . '<cell>', $row) . '</cell>' . PHP_EOL .
                '</row>' . PHP_EOL;
        }
        $result .= '</rows>';

        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testXml2Output()
    {
        $this->object = new Exporter(DriverFactory::buildXML2());
        list($header, $data) = $this->createDataTests();

        $result = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $result .= '<offers>' . PHP_EOL;
        $_data = $data;
        foreach ($_data as $row) {
            $result .= '<offer>' . PHP_EOL;
            foreach ($row as $key => $item) {
                if (is_integer($key)) {
                    $key = $item;
                }
                $result .= '<' . $key . '>' . $item . '</' . $key . '>' . PHP_EOL;
            }
            $result .= '</offer>' . PHP_EOL;
        }
        $result .= '</offers>';


        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testHtmlOutput()
    {
        $this->object = new Exporter(DriverFactory::buildHtml());
        list($header, $data) = $this->createDataTests();

        $result = '<table class="table-csv-data">' . PHP_EOL;
        $result .= '<tbody>' . PHP_EOL;

        $_data = array_merge([$header], $data);
        foreach ($_data as $row) {
            $result .= '<tr>' . PHP_EOL;
            foreach ($row as $key => $item) {
                $result .= '<td>' . $item . '</td>' . PHP_EOL;
            }
            $result .= '</tr>' . PHP_EOL;
        }
        $result .= '</tbody>' . PHP_EOL;
        $result .= '</table>';


        $this->assertEquals($result, $this->getOutput());
    }

    /**
     * @return array
     * @throws \Exception
     */
    private function createDataTests(): array
    {
        $header = [
            'name', 'email'
        ];
        $data = [];
        $this->object->setHeader($header);

        $max = random_int(1, 1);

        for ($i = 0; $i < $max; $i++) {
            $row = [
                'name' => 'Вася Питериский',
                'email' => 'vacia@mail.ru'
            ];
            $this->object->addRow($row);
            $data [] = $row;
        }

        return [
            $header, $data
        ];
    }

    /**
     * @return string
     */
    private function getOutput(): string
    {
        $fromMethod = debug_backtrace()[1]['function'];
        switch (true) {
            case stripos($fromMethod, 'csv') !== false:
                $ext = 'csv';
                break;
            case stripos($fromMethod, 'txt') !== false:
                $ext = 'txt';
                break;
            case stripos($fromMethod, 'xml') !== false:
                $ext = 'xml';
                break;
            case stripos($fromMethod, 'html') !== false:
                $ext = 'html';
                break;
        }

        $fileName = 'test.' . $ext;

        ob_start();
        $this->object->output($fileName);
        return ob_get_clean();
    }

    /**
     * @param array $data
     * @param string $delimiter
     * @param string $enclosure
     * @param bool $strongEnclosure
     * @return string
     */
    private function makeCsv(array $data, string $delimiter, string $enclosure, bool $strongEnclosure = false, bool $asFormulas = false): string
    {
        $buffer = fopen('php://temp', 'r+');
        $csv = '';

        if($asFormulas===true){
            $strongEnclosure = $asFormulas;
        }

        if($strongEnclosure===false){
            foreach ($data as $fields) {
                fputcsv($buffer, $fields, $delimiter, $enclosure);
            }
            rewind($buffer);
            while (!feof($buffer)) {
                $line = fgets($buffer);
                $csv .= $line;
            }
            fclose($buffer);

            return $csv;
        }

        foreach ($data as $fields) {
            array_walk($fields,function(&$v) use ($enclosure,$asFormulas){
                $pref = $asFormulas===true?'=':'';
                $v =  $pref . $enclosure . trim($v) .$enclosure;
            });

            $line = implode($delimiter,$fields) . PHP_EOL;

            $csv .= $line;
        }


        return $csv;
    }
}
