<?php

namespace ExportEngine\Drivers;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use ExportEngine\Contracts\ExportProcessInterface;
use PhpOffice\PhpSpreadsheet\Writer\IWriter;

/**
 * экспорт Xlsx
 *
 * @author d.lanec
 */
class Xlsx implements ExportProcessInterface
{

    use SourceTrait;
    use PrepareValueTrait;

    protected $charSet;
    protected $data = [];
    protected $filename;
    protected $exportApi;
    protected $useXLSXWriter;
    protected $XLSXWriter;
    protected $head = [];
    protected $ext = 'xlsx';
    protected $sheet;
    protected $spreadsheet;

    /**
     * Xlsx constructor.
     * @throws \PhpOffice\PhpSpreadsheet\Exception
     */
    public function __construct()
    {

        $this->spreadsheet = new Spreadsheet();

        $this->sheet = $this->spreadsheet->getActiveSheet();
    }

    /**
     * @return string
     */
    public function getExtendsion(): string
    {
        return $this->ext;
    }

    /**
     * @return array
     */
    public function getHead(): array
    {
        return $this->head;
    }

    /**
     * @param array $head
     * @return $this
     */
    public function setHead(array $head)
    {
        $this->head = $head;
        return $this;
    }

    /**
     * @throws \PhpOffice\PhpSpreadsheet\Writer\Exception
     */
    public function output(): void
    {

        $writer = $this->createWriter();

        $filename = filter_var($this->fileName, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
        header('Content-Transfer-Encoding: binary');
        header('Content-Type: application/vnd.ms-excel');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Content-type: application/vnd.ms-excel; charset=' . $this->charSet);

        $writer->save('php://output');
    }

    /**
     * @return IWriter
     */
    protected function createWriter(): IWriter
    {
        $this->fillSheetData();
        return (new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($this->spreadsheet));
    }

    /**
     * Заполняет данными лист
     * @param int|null $limit
     */
    final protected function fillSheetData(int $limit = null): void
    {
        $chars = range('A', 'Z');

        foreach ($this->head as $value) {
            $this
                ->sheet
                ->setCellValue(current($chars) . 1, $value);
            next($chars);
        }

        reset($chars);
        $i = 2;
        $this->source->clearPointer();
        while ($row = $this->source->walk()) {
            foreach ($row as $k => $value) {

                $cell = current($chars). $i ;

              //  dump($cell. ' => '. $this->prepareValue($value)  );

                $this
                    ->sheet
                    ->setCellValue($cell , $this->prepareValue($value) );
                next($chars);
            }
            $i++;
            reset($chars);

            if($limit && $i >= $limit){
                break;
            }
        }
    }

    /**
     * @param string $file
     * @throws \PhpOffice\PhpSpreadsheet\Writer\Exception
     */
    public function exportToFile(string $file): void
    {
        $this->createWriter()->setPreCalculateFormulas(false)->save($file);
    }


    /**
     * @param string $charSet
     * @return $this
     */
    public function setCharSet(string $charSet)
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @param array $data
     * @return $this
     */
    public function setData(array $data)
    {
        $this->data = $data;
        return $this;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setFileName(string $name)
    {
        $this->filename = $name;
        return $this;
    }
}
