<?php

namespace ExportEngine\Drivers;


use ExportEngine\Exceptions\ExportException;

/**
 * Description of Csv
 *
 * @author d.lanec
 */
class Csv implements \ExportEngine\Contracts\ExportProcessInterface
{

    use SourceTrait;
    use PrepareValueTrait;

    protected $file;
    protected $data;
    protected $separator = ';';
    protected $enclosure = '"';
    protected $escape = '\\';
    protected $linebreak = "\r\n";
    protected $fileName;
    protected $charSet;
    protected $head;
    protected $fp;
    private static $ext = 'csv';
    protected $asАormulas = false;

    /**
     * Csv constructor.
     * @param \SplFileObject $file
     */
    public function __construct(\SplFileObject $file)
    {
        $this->file = $file;
    }

    /**
     * @param bool $asАormulas
     */
    public function setAsАormulas(bool $asАormulas)
    {
        $this->asАormulas = $asАormulas;
    }


    /**
     * @param array $head
     * @return $this
     */
    public function setHead(array $head)
    {
        $this->head = $head;
        return $this;
    }

    /**
     * @param string $charSet
     * @return $this
     */
    public function setCharSet(string $charSet)
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @return string
     */
    public function getExtendsion(): string
    {
        return self::$ext;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setFileName(string $name)
    {
        $this->fileName = $name;
        return $this;
    }

    /**
     * @param string $separator
     * @return $this
     */
    public function setSeparator(string $separator)
    {
        $this->separator = $separator;
        return $this;
    }

    /**
     * @param string $enclosure
     * @return $this
     */
    public function setEnclosure(string $enclosure)
    {
        $this->enclosure = $enclosure;
        return $this;
    }

    /**
     * @param string $escape
     * @return $this
     */
    public function setEscape(string $escape)
    {
        $this->escape = $escape;
        return $this;
    }

    /**
     * @param string $linebreak
     * @return $this
     */
    public function setLinebreak(string $linebreak)
    {
        $this->linebreak = $linebreak;
        return $this;
    }

    /**
     *
     */
    public function output(): void
    {
        $filename = filter_var($this->fileName, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
        header('Content-Transfer-Encoding: binary');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Content-Type: text/csv; charset=' . $this->charSet);
        $this->outputOnlyData();
    }

    /**
     * вывод чистых данных в буфер без заголовка
     */
    protected function outputOnlyData(): void
    {
        echo $this->getFirstLine();
        $this->source->clearPointer();

        while ($row = $this->source->walk()) {
            echo $this->pepareRow($row);
        }
        echo $this->getLastLine();
    }


    /**
     * @return string
     */
    protected function getFirstLine(): string
    {
        if (count($this->head)) {
            return $this->makeCsv($this->head);
        }

        return '';
    }

    /**
     * обработка строки массива
     * @param array $row
     * @return string
     */
    protected function pepareRow(array $row): string
    {
        return $this->makeCsv($row);
    }

    /**
     * @param array $rows
     * @return string
     */
    private function makeCsv(array $rows): string
    {
        $csv = '';
        $enc = $this->enclosure;
        array_walk($rows, function (&$value) {
            $value = trim($value);
        });
        $form = $this->asАormulas === true ? '=' : '';

        foreach ($rows as $row) {

            $csv .= $form . $enc . $this->prepareValue($row) . $enc . $this->separator;
        }

        return rtrim($csv, $this->separator) . PHP_EOL;
    }



    /**
     * @return string
     */
    protected function getLastLine(): string
    {
        return '';
    }


    /**
     * @param string $path
     * @throws ExportException
     */
    public function exportToFile(string $path): void
    {

        if (!$outputHandle = fopen($path, 'w+')) {
            throw new ExportException('Unable to open file to write!');
        }

        fwrite($outputHandle, $this->getFirstLine());

        $this->source->clearPointer();

        while ($row = $this->source->walk()) {
            fwrite($outputHandle, $this->pepareRow($row));
        }

        fwrite($outputHandle, $this->getLastLine());

        fflush($outputHandle);

        fclose($outputHandle);
    }

    /**
     * @param array $data
     * @return $this
     */
    public function setData(array $data)
    {
        $this->data = $data;
        return $this;
    }


}