<?php

namespace ExportEngine;

use ExportEngine\Storage\SourceData;

/**
 * Движек экспорта данных в различные форматы
 *
 * 1. Позволяет экспортировать в csv,xml,xlsx,xls
 * 2. Данные вставляет как целым массивом так и построчно
 * 3. Вывод в файл на сервере так и на скачку в браузер
 *
 * @author d.lanec
 */
class Exporter
{

    protected $driver;
    protected $data = [];
    protected $header = [];
    protected $fileName;
    protected $charSet = 'utf-8';
    private $source;
    private static $limitFlushMemory = 10;

    /**
     * @var float|null
     */
    private $memory;

    /**
     * Exporter constructor.
     * @param Contracts\ExportProcessInterface $driver
     */
    public function __construct(Contracts\ExportProcessInterface $driver)
    {
        $this->driver = $driver;
        //$this->fileName = sprintf('tempfile.%s', $this->driver->getExtendsion());
        $this->memory = $this->getMemoryUsage();
        $this->source = new SourceData(tmpfile());
        $this->fileName = $this->source->getSourcePath();
    }

    private function getMemoryUsage(): float
    {
        return round(memory_get_usage() / 1024 / 1024, 2);
    }

    /**
     * @param array $row
     * @return Exporter
     */
    final public function addRow(array $row): self
    {
        $this->data[] = $row;

        if(($this->getMemoryUsage() - $this->memory) > self::$limitFlushMemory){
            $this->flushData();
        }

        return $this;
    }


    /**
     * Сброс данных на диск
     */

    final public function flushData()
    {
        $this->source->addData($this->data);
        $this->data = [];
    }

    /**
     * @param string $charSet
     * @return $this
     */
    final public function setCharSet(string $charSet): self
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @param string $filename
     * @return $this
     */
    final public function setFileName(string $filename): self
    {
        $this->fileName = $filename;
        return $this;
    }

    /**
     * @param array $data
     * @return Exporter
     */
    final public function addData(array $data): self
    {
        $this->data = array_merge($data, $this->data);
        return $this;
    }



    /**
     * @param array $header
     * @return Exporter
     */
    final public function setHeader(array $header): self
    {
        $this->header = $header;
        return $this;
    }

    /**
     * @param array $data
     * @return Exporter
     */
    final public function setData(array $data): self
    {
        $this->data = $data;
        return $this;
    }

    /**
     * Вывод в буфер содержимого файла
     */
    final public function output(): void
    {
        $this->flushData();
        $this
            ->driver
            ->setCharSet($this->charSet)
            ->setFileName($this->fileName)
            ->setHead($this->header)
            ->setSourceData($this->source)
            ->output();
    }

    /**
     * @param string $path
     */
    final public function exportToFile(string $path): void
    {

        $this->flushData();
        $this
            ->driver
            ->setCharSet($this->charSet)
            ->setFileName($this->fileName)
            ->setHead($this->header)
            ->setSourceData($this->source)
            ->exportToFile($path);
    }




    /**
     * фильтрует данные и осталяет только те которые определены шапкой
     * @param array $data
     * @param array $header
     * @param null $userFn
     * @return array
     */
    final public function prepareData(array $data, array $header, $userFn = null): array
    {

        if (count($header) === 0) {
            return $data;
        }

        $new = [];

        if (!$userFn) {
            $userFn = function ($v, $s, $header) use (&$new) {

                $item = [];
                foreach ($header as $key) {
                    $item[$key] = $v[$key];
                }
                $new [] = $item;
            };
        }

        $res = array_walk($data, $userFn, $header);


        return $new;
    }



    /**
     * @return string
     */
    final public function getFileName(): string
    {
        return $this->fileName;
    }

}