<?php

namespace ExportEngine;

use PHPUnit\Framework\TestCase;
use ExportEngine\Factories\DriverFactory;

class ExporterTest extends TestCase
{

    /**
     * @var Exporter
     */
    protected $object;

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     */
    protected function setUp()
    {
        $this->object = new Exporter(DriverFactory::buildCsv());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutput()
    {
        list($header, $data) = $this->createDataTests();

        $result =
            $this
                ->makeCsv([$header], ';', '"') .
            $this
                ->makeCsv($data, ';', '"');

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutputSeparator()
    {
        $this->object = new Exporter(DriverFactory::buildCsv()->setSeparator(','));
        list($header, $data) = $this->createDataTests();

        $result =
            $this
                ->makeCsv([$header], ',', '"') .
            $this
                ->makeCsv($data, ',', '"');

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testCsvOutputEnclosure()
    {
        $this->object = new Exporter(DriverFactory::buildCsv()->setEnclosure("'"));
        list($header, $data) = $this->createDataTests();

        $result =
            $this
                ->makeCsv([$header], ';', "'") .
            $this
                ->makeCsv($data, ';', "'");

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testTextOutputEnclosure()
    {
        $this->object = new Exporter(DriverFactory::buildTxt()->setEnclosure("'"));
        list($header, $data) = $this->createDataTests();

        $result =
            $this
                ->makeCsv([$header], ';', "'") .
            $this
                ->makeCsv($data, ';', "'");

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @return array
     * @throws \Exception
     */
    private function createDataTests(): array
    {
        $header = [
            'name', 'email'
        ];
        $data = [];
        $this->object->setHeader($header);

        $max = random_int(10, 20);
        for ($i = 0; $i < $max; $i++) {
            $row = [
                'name' => 'Вася Питериский',
                'email' => 'vacia@mail.ru'
            ];
            $this->object->addRow($row);
            $data [] = $row;
        }
        return [
            $header, $data
        ];
    }


    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testXmlOutput()
    {
        $this->object = new Exporter(DriverFactory::buildXML());
        list($header, $data) = $this->createDataTests();

        $result = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;

        $_data = array_merge([$header], $data);
        foreach ($_data as $row) {
            $result .= '<row><cell>' . implode('</cell><cell>', $row) . '</cell></row>';
        }

        $result .= PHP_EOL;

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testXml2Output()
    {
        $this->object = new Exporter(DriverFactory::buildXML2());
        list($header, $data) = $this->createDataTests();

        $result = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $result .= '<offers>';
        $_data = array_merge([$header], $data);
        foreach ($_data as $row) {
            $result .= '<offer>';
            foreach ($row as $key => $item) {
                if (is_integer($key)) {
                    $key = $item;
                }
                $result .= '<' . $key . '>' . $item . '</' . $key . '>';
            }
            $result .= '</offer>';
        }
        $result .= '</offers>';
        $result .= PHP_EOL;

        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @throws \Exception
     * @covers \ExportEngine\Exporter::output
     */
    public function testHtmlOutput()
    {
        $this->object = new Exporter(DriverFactory::buildHtml());
        list($header, $data) = $this->createDataTests();

        $result = '<table class="table-csv-data">' . PHP_EOL;

        $_data = array_merge([$header], $data);
        foreach ($_data as $row) {
            $result .= '<tr>' . PHP_EOL;
            foreach ($row as $key => $item) {
                $result .= '<td>' . $item . '</td>' . PHP_EOL;
            }
            $result .= '</tr>' . PHP_EOL;
        }
        $result .= '</table>';


        $this->assertEquals($result, $this->object->output());
    }

    /**
     * @param array $values
     * @param string $separator
     * @param string $enclosure
     * @return string
     */
    private function makeCsv(array $values, string $separator, string $enclosure): string
    {
        $csv = '';
        $enc = $enclosure;
        foreach ($values as $row) {
            $csv .= $enc . implode($enc . $separator . $enc, $row) . $enc . PHP_EOL;
        }
        return $csv;
    }
}
