<?

namespace ExportEngine\Drivers;

use League\Csv\Writer;

/**
 * Description of Csv
 *
 * @author d.lanec
 */
class Csv implements \ExportEngine\Contracts\ExportProcessInterface
{

    protected $file;
    protected $data;
    protected $separator = ';';
    protected $enclosure = '"';
    protected $escape = '\\';
    protected $linebreak = "\r\n";
    protected $fileName;
    protected $charSet;
    protected $head;
    protected $fp;
    private static $ext = 'csv';

    /**
     * Csv constructor.
     * @param \SplFileObject $file
     */
    public function __construct(\SplFileObject $file)
    {
        $this->file = $file;
    }

    /**
     * @param array $head
     * @return $this
     */
    public function setHead(array $head)
    {
        $this->head = $head;
        return $this;
    }

    /**
     * @param string $charSet
     * @return $this
     */
    public function setCharSet(string $charSet)
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @return string
     */
    public function getExtendsion(): string
    {
        return self::$ext;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setFileName(string $name)
    {
        $this->fileName = $name;
        return $this;
    }

    /**
     * @param string $separator
     * @return $this
     */
    public function setSeparator(string $separator)
    {
        $this->separator = $separator;
        return $this;
    }

    /**
     * @param string $enclosure
     * @return $this
     */
    public function setEnclosure(string $enclosure)
    {
        $this->enclosure = $enclosure;
        return $this;
    }

    /**
     * @param string $escape
     * @return $this
     */
    public function setEscape(string $escape)
    {
        $this->escape = $escape;
        return $this;
    }

    /**
     * @param string $linebreak
     * @return $this
     */
    public function setLinebreak(string $linebreak)
    {
        $this->linebreak = $linebreak;
        return $this;
    }

    /**
     *
     * @return string
     */
    public function output(): string
    {
        $filename = filter_var($this->fileName, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW);
        header('Content-Transfer-Encoding: binary');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Content-Type: text/csv; charset=' . $this->charSet);
        return $this->getContentString();
    }

    /**
     * @param string $path
     * @param int $flag
     */
    public function exportToFile(string $path, int $flag = 0): void
    {
        if (!$content = $this->getContentString()) {
            return;
        }

        if (!$this->fp) {
            $this->fp = fopen($path, 'w+');

            @flock($this->fp, LOCK_SH);
        }

        if (fwrite($this->fp, $content) === false) {
            throw new \RuntimeException('write file error ' . $path);
        }
    }

    /**
     * @param array $data
     * @return $this
     */
    public function setData(array $data)
    {
        $this->data = $data;
        return $this;
    }

    /**
     * @return string
     */
    protected function getContentString(): string
    {
        $content = $this->csvToString();
        if ($this->charSet === 'windows-1251') {
            $content = iconv('utf-8', $this->charSet . '//TRANSLIT//IGNORE', $content);
        }
        return $content;
    }


    /**
     * @return string
     */
    private function csvToString(): string
    {

        if (count($this->head)) {
            $csv = $this->makeCsv([$this->head]);
        }

        $csv .= $this->makeCsv($this->data);

        return $csv;
    }

    /**
     * @param array $values
     * @return string
     */
    private function makeCsv(array $values): string
    {
        $csv = '';
        $enc = $this->enclosure;
        foreach ($values as $row) {
            $csv .= $enc . implode($enc . $this->separator . $enc, $row) . $enc . PHP_EOL;
        }
        return $csv;
    }

    /**
     * @return Writer
     */
    protected function createWriter(): Writer
    {

        $writer = Writer::createFromStream(tmpfile());

        $writer->setDelimiter($this->separator);

        $writer->setEscape($this->escape);
        $writer->setNewline($this->linebreak);


        //небольшой костыль для принудительного экранирование значений
        $writer->setEnclosure($this->enclosure);

        if ($this->charSet === 'utf-8') {
            $writer->setOutputBOM(Writer::BOM_UTF8); //adding the BOM sequence on output
        }

        if (!empty($this->head)) {
            $writer->insertOne($this->head);
        }

        $writer->insertAll($this->data);

        return $writer;
    }

    public function __destruct()
    {
        if ($this->fp) {
            flock($this->fp, LOCK_UN);
            fclose($this->fp);
        }
    }

}