<?php

namespace ExportEngine;

/**
 * Движек экспорта данных в различные форматы
 *
 * 1. Позволяет экспортировать в csv,xml,xlsx,xls
 * 2. Данные вставляет как целым массивом так и построчно
 * 3. Вывод в файл на сервере так и на скачку в браузер
 *
 * @author d.lanec
 */
class Exporter
{

    protected $driver;
    protected $data = [];
    protected $header = [];
    protected $fileName;
    protected $charSet = 'utf-8';

    /**
     * Exporter constructor.
     * @param Contracts\ExportProcessInterface $driver
     */
    public function __construct(Contracts\ExportProcessInterface $driver)
    {
        $this->driver = $driver;
        $this->fileName = sprintf('tempfile.%s', $this->driver->getExtendsion());
    }

    /**
     * @param string $charSet
     * @return $this
     */
    final public function setCharSet(string $charSet): self
    {
        $this->charSet = $charSet;
        return $this;
    }

    /**
     * @param string $filename
     * @return $this
     */
    final public function setFileName(string $filename): self
    {
        $this->fileName = $filename;
        return $this;
    }

    /**
     * @param array $data
     * @return Exporter
     */
    final public function addData(array $data): self
    {
        $this->data = array_merge($data, $this->data);
        return $this;
    }

    /**
     * @param array $row
     * @return Exporter
     */
    final public function addRow(array $row): self
    {
        $this->data[] = $row;
        return $this;
    }

    /**
     * @param array $header
     * @return Exporter
     */
    final public function setHeader(array $header): self
    {
        $this->header = $header;
        return $this;
    }

    /**
     * @param array $data
     * @return Exporter
     */
    final public function setData(array $data): self
    {
        $this->data = $data;
        return $this;
    }

    /**
     * @return string
     */
    final public function output(): string
    {
        return $this
            ->driver
            ->setCharSet($this->charSet)
            ->setFileName($this->fileName)
            ->setHead($this->header)
            ->setData($this->data)
            ->output();
    }

    /**
     * фильтрует данные и осталяет только те которые определены шапкой
     * @param array $data
     * @param array $header
     * @param null $userFn
     * @return array
     */
    final public function prepareData(array $data, array $header, $userFn = null): array
    {

        if (count($header) === 0) {
            return $data;
        }

        $new = [];

        if (!$userFn) {
            $userFn = function ($v, $s, $header) use (&$new) {

                $item = [];
                foreach ($header as $key) {
                    $item[$key] = $v[$key];
                }
                $new [] = $item;
            };
        }

        $res = array_walk($data, $userFn, $header);


        return $new;
    }

    /**
     * @param string $path
     * @param int $flag
     */
    final public function exportToFile(string $path,int $flag = 0): void
    {

        $this
            ->driver
            ->setCharSet($this->charSet)
            ->setFileName($this->getFileName())
            ->setHead($this->header)
            ->setData($this->data)
            ->exportToFile($path, $flag);

        $this->data = [];
        $this->driver->setData([]);
    }

    /**
     * @return string
     */
    final public function getFileName():string
    {
        return $this->fileName;
    }

}