<?php

namespace ExportEngine\Helpers;

use PhpOffice\PhpSpreadsheet\Calculation\Calculation;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Writer\BaseWriter;
use PhpOffice\PhpSpreadsheet\Writer\Exception as WriterException;

/**
 * Class XmlWriter
 *
 * @author Aleksei Kuznetsov
 */
class XmlWriter extends BaseWriter
{
    /**
     * Spreadsheet object.
     *
     * @var Spreadsheet
     */
    protected $spreadsheet;

    /**
     * Create a new XML.
     *
     * @param Spreadsheet $spreadsheet
     */
    public function __construct(Spreadsheet $spreadsheet)
    {
        $this->spreadsheet = $spreadsheet;
    }

    /**
     * Save PhpSpreadsheet to file.
     *
     * @param string $pFilename Name of the file to save
     *
     * @throws WriterException
     */
    public function save($pFilename): void
    {
        // garbage collect
        $this->spreadsheet->garbageCollect();

        $saveDebugLog = Calculation::getInstance($this->spreadsheet)->getDebugLog()->getWriteDebugLog();
        Calculation::getInstance($this->spreadsheet)->getDebugLog()->setWriteDebugLog(false);
        $saveArrayReturnType = Calculation::getArrayReturnType();
        Calculation::setArrayReturnType(Calculation::RETURN_ARRAY_AS_VALUE);

        // Open file
        $fileHandle = fopen($pFilename, 'wb+');
        if ($fileHandle === false) {
            throw new WriterException("Could not open file $pFilename for writing.");
        }

        // Write headers
        fwrite($fileHandle, $this->generateXMLHeader());

        // Write data
        fwrite($fileHandle, $this->generateSheetData());

        // Write footer
        fwrite($fileHandle, $this->generateXMLFooter());

        // Close file
        fclose($fileHandle);

        Calculation::setArrayReturnType($saveArrayReturnType);
        Calculation::getInstance($this->spreadsheet)->getDebugLog()->setWriteDebugLog($saveDebugLog);
    }

    /**
     * @return string
     */
    public function generateXMLHeader(): string
    {
        $content = '<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL;
        $content .= '<!DOCTYPE csv>' . PHP_EOL;
        $content .= '<csv>' . PHP_EOL;

        return $content;
    }

    /**
     * @return string
     */
    public function generateSheetData(): string
    {
        // Fetch sheets
        $sheets = $this->spreadsheet->getAllSheets();

        // Construct XML
        $content = '';

        foreach ($sheets as $sheet) {
            // Get worksheet dimension
            [$firstColumnAndRow, $highestColumnAndRow] = explode(':', $sheet->calculateWorksheetDimension());

            [$firstColumn, $firstRow] = $this->getColumnIndexAndRowIndexFromString($firstColumnAndRow);
            [$highestColumn, $highestRow] = $this->getColumnIndexAndRowIndexFromString($highestColumnAndRow);

            for ($row = $firstRow; $row <= $highestRow; $row++) {
                // Start a new rowData
                $rowData = [];

                for ($column = $firstColumn; $column <= $highestColumn; $column++) {
                    if ($sheet->cellExistsByColumnAndRow($column, $row)) {
                        $rowData[$column] = Coordinate::stringFromColumnIndex($column) . $row;
                    } else {
                        $rowData[$column] = '';
                    }
                }

                $content .= $this->generateRow($sheet, $rowData);
            }
        }

        return $content;
    }

    /**
     * @return string
     */
    public function generateXMLFooter(): string
    {
        $content = '';
        $content .= '</csv>' . PHP_EOL;

        return $content;
    }

    /**
     * @param string $coordinate
     *
     * @return array
     * @throws \PhpOffice\PhpSpreadsheet\Exception
     */
    private function getColumnIndexAndRowIndexFromString(string $coordinate): array
    {
        [$column, $row] = Coordinate::coordinateFromString($coordinate);

        return [
            Coordinate::columnIndexFromString($column),
            (int)$row,
        ];
    }

    /**
     * @param Worksheet $sheet
     * @param array $rowData
     *
     * @return string
     */
    private function generateRow(Worksheet $sheet, array $rowData): string
    {
        $content = '    <row>' . PHP_EOL;

        foreach ($rowData as $cellValue) {
            $cell = ($cellValue > '') ? $sheet->getCell($cellValue) : '';

            if ($cell instanceof Cell) {
                $cellData = NumberFormat::toFormattedString(
                    $cell->getValue(),
                    $sheet->getParent()->getCellXfByIndex($cell->getXfIndex())->getNumberFormat()->getFormatCode(),
                    [$this, 'formatColor']
                );

                $cellData = htmlspecialchars($cellData);
                // Converts the cell content so that spaces occuring at beginning of each new line are replaced.
                // Example: "  Hello\n to the world" is converted to "&nbsp;&nbsp;Hello\n&nbsp;to the world"
                $cellData = preg_replace('/(?m)(?:^|\\G) /', '&nbsp;', $cellData);
                // convert newline "\n" to '<br>'
                $cellData = nl2br($cellData);
            } else {
                $cellData = '&nbsp;';
            }

            $content .= '        <cell>' . $cellData . '</cell>' . PHP_EOL;
        }

        $content .= '    </row>' . PHP_EOL;

        return $content;
    }
}
