<?php

namespace ExportEngine\Factories;

use ExportEngine\Concrete\Drivers\CsvDriver;
use ExportEngine\Concrete\Drivers\HtmlDriver;
use ExportEngine\Concrete\Drivers\TxtDriver;
use ExportEngine\Concrete\Drivers\XlsDriver;
use ExportEngine\Concrete\Drivers\XlsxDriver;
use ExportEngine\Concrete\Drivers\XmlDriver;
use ExportEngine\Contracts\DriverInterface;
use ExportEngine\Contracts\FileConfiguratorInterface;
use ExportEngine\Contracts\FileLinesConfiguratorInterface;
use ExportEngine\Drivers\Csv;
use ExportEngine\Drivers\Html;
use ExportEngine\Drivers\Txt;
use ExportEngine\Drivers\Xls;
use ExportEngine\Drivers\Xlsx;
use ExportEngine\Drivers\Xml;
use ExportEngine\Exceptions\FileFormatNotSupportedException;
use ExportEngine\Contracts\ExportProcessInterface;

/**
 * Фабрика драйверов
 *
 * @author d.lanec
 */
class DriverFactory
{
    /**
     * Создание объекта по типу
     *
     * @param string $fileType
     * @param FileConfiguratorInterface|FileLinesConfiguratorInterface $fileConfig
     *
     * @return DriverInterface
     * @throws FileFormatNotSupportedException
     */
    public static function buildByFileType(string $fileType, FileConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        $methodName = 'build' . ucfirst(strtolower($fileType));

        if (!method_exists(self::class, $methodName)) {
            throw new FileFormatNotSupportedException($fileType);
        }

        return self::$methodName($fileConfig);
    }

    /**
     * @param FileLinesConfiguratorInterface $fileConfig
     *
     * @return CsvDriver
     */
    public static function buildCsv(FileLinesConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Csv($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return HtmlDriver
     */
    public static function buildHtml(FileConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Html($fileConfig);
    }

    /**
     * @param FileLinesConfiguratorInterface $fileConfig
     *
     * @return TxtDriver
     */
    public static function buildTxt(FileLinesConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Txt($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XlsDriver
     */
    public static function buildXls(FileConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Xls($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XlsxDriver
     */
    public static function buildXlsx(FileConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Xlsx($fileConfig);
    }

    /**
     * @param FileConfiguratorInterface $fileConfig
     *
     * @return XmlDriver
     */
    public static function buildXml(FileConfiguratorInterface $fileConfig): ExportProcessInterface
    {
        return new Xml($fileConfig);
    }
}
